#ifndef CHARSET_MAP_H_031020121518
#define CHARSET_MAP_H_031020121518

#include <mail_getter/recognizerWrapper.h>
#include <mail_getter/charset_converter.h>
#include <map>
#include <string>
#include <boost/algorithm/string.hpp>
#include <sstream>

struct AddAliasForUninitializedCharset : std::logic_error {
    AddAliasForUninitializedCharset(const std::string & aliasCharset)
        : std::logic_error("Trying add alias " + aliasCharset
            + " for uninitialized charset") {}
};

class CharsetTraitsMap {
public:
    typedef Recognizer::CodesEnum CodesEnum;

    struct CharsetTraits {
        std::string charset;
        CodesEnum code;
        const TextConverter & convertor;
        CharsetTraits(CodesEnum code, const std::string & charset,
                const TextConverter & convertor )
        : charset(charset), code(code), convertor(convertor) {}
    };

    CharsetTraitsMap()
    : defaultTraits(Recognizer::CODES_WIN, "CP1251", normalConverter) {
        using namespace Recognizer;
        addItem(CODES_WIN, "CP1251");
        addItem(CODES_KOI8, "KOI8-R");
        addItem(CODES_ALT, "IBM866");
        addItem(CODES_MAC, "MAC-CYRILLIC");
        addItem(CODES_MAIN, "ISO-8859-5");
        addItem(CODES_ASCII, "CP1252");
        addItem(CODES_WIN_EAST, "CP1250");
        addItem(CODES_ISO_EAST, "ISO-8859-2");
        addItem(CODES_WINWIN, "WINWIN", winWinConverter);
        addItem(CODES_KOIKOI, "KOIKOI", koiKoiConverter);
        addItem(CODES_IBM855, "IBM-855");
        addItem(CODES_UTF8, "UTF-8", utfConverter);
        addItem(CODES_KAZWIN, "CP1251");
        addItem(CODES_TATWIN, "CP1251");
        addItem(CODES_ARMSCII, "ARMSCII-8");
        addItem(CODES_GEO_ITA, "GEORGIAN-ACADEMY");
        addItem(CODES_GEO_PS, "GEORGIAN-PS");
        addItem(CODES_ISO_8859_3, "ISO-8859-3");
        addItem(CODES_ISO_8859_4, "ISO-8859-4");
        addItem(CODES_ISO_8859_6, "ISO-8859-6");
        addItem(CODES_ISO_8859_7, "ISO-8859-7");
        addItem(CODES_ISO_8859_8, "ISO-8859-8");
        addItem(CODES_ISO_8859_9, "ISO-8859-9");
        addItem(CODES_ISO_8859_13, "ISO-8859-13");
        addItem(CODES_ISO_8859_15, "ISO-8859-15");
        addItem(CODES_ISO_8859_16, "ISO-8859-16");
        addItem(CODES_WINDOWS_1253, "CP1253");
        addItem(CODES_WINDOWS_1254, "CP1254");
        addItem(CODES_WINDOWS_1255, "CP1255");
        addItem(CODES_WINDOWS_1256, "CP1256");
        addItem(CODES_WINDOWS_1257, "CP1257");
        addItem(CODES_CP1046, "CP1046");
        addItem(CODES_CP1124, "CP1124");
        addItem(CODES_CP1125, "CP1125");
        addItem(CODES_CP1129, "CP1129");
        addItem(CODES_CP1131, "CP1131");
        addItem(CODES_CP1133, "CP1133");
        addItem(CODES_CP1161, "CP1161");
        addItem(CODES_CP1162, "CP1162");
        addItem(CODES_CP1163, "CP1163");
        addItem(CODES_CP1258, "CP1258");
        addItem(CODES_CP437, "CP437");
        addItem(CODES_CP737, "CP737");
        addItem(CODES_CP775, "CP775");
        addItem(CODES_CP850, "CP850");
        addItem(CODES_CP852, "CP852");
        addItem(CODES_CP853, "CP853");
        addItem(CODES_CP856, "CP856");
        addItem(CODES_CP857, "CP857");
        addItem(CODES_CP858, "CP858");
        addItem(CODES_CP860, "CP860");
        addItem(CODES_CP861, "CP861");
        addItem(CODES_CP862, "CP862");
        addItem(CODES_CP863, "CP863");
        addItem(CODES_CP864, "CP864");
        addItem(CODES_CP865, "CP865");
        addItem(CODES_CP869, "CP869");
        addItem(CODES_CP874, "CP874");
        addItem(CODES_CP922, "CP922");
        addItem(CODES_ISO646_CN, "ISO646-CN");
        addItem(CODES_ISO646_JP, "ISO646-JP");
        addItem(CODES_ISO8859_10, "ISO8859-10");
        addItem(CODES_ISO8859_11, "ISO8859-11");
        addItem(CODES_ISO8859_14, "ISO8859-14");
        addItem(CODES_KOI8_T, "KOI8-T" );
        addItem(CODES_PT154, "PT154" );
        addItem(CODES_RK1048, "RK1048" );
        addItem(CODES_TCVN, "TCVN" );
        addItem(CODES_TIS620, "TIS620" );
        addItem(CODES_VISCII, "VISCII" );
        addItem(CODES_BIG5, "BIG5" );
        addItem(CODES_BIG5_HKSCS, "BIG5-HKSCS" );
        addItem(CODES_CP932, "CP932" );
        addItem(CODES_CP936, "CP936" );
        addItem(CODES_CP949, "CP949" );
        addItem(CODES_CP950, "CP950" );
        addItem(CODES_EUC_CN, "EUC-CN" );
        addItem(CODES_EUC_JP, "EUC-JP" );
        addItem(CODES_EUC_KR, "EUC-KR" );
        addItem(CODES_EUC_TW, "EUC-TW" );
        addItem(CODES_GB18030, "GB18030" );
        addItem(CODES_GBK, "GBK" );
        addItem(CODES_ISO_2022_CN, "ISO-2022-CN" );
        addItem(CODES_ISO_2022_CN_EXT, "ISO-2022-CN-EXT" );
        addItem(CODES_ISO_2022_JP, "ISO-2022-JP" );
        addItem(CODES_ISO_2022_JP_2, "ISO-2022-JP-2" );
        addItem(CODES_ISO_2022_KR, "ISO-2022-KR" );
        addItem(CODES_JOHAB, "JOHAB" );
        addItem(CODES_SHIFT_JIS, "SHIFT-JIS" );
        addItem(CODES_UTF16, "UTF-16" );
        addAlias(CODES_KS_C_5601_1987, "KS_C_5601-1987", CODES_CP949 );
    }

    const CharsetTraits & getCharsetTraits( CodesEnum code ) const {
        CodeToTraits::const_iterator i(code2Traits.find(code));
        return i != code2Traits.end() ? i->second : defaultTraits;
    }

    CharsetTraits getCharsetTraits( const std::string & charset ) const {
        CharsetToTraits::const_iterator i(charset2Traits.find(boost::to_upper_copy(charset)));
        if( i != charset2Traits.end() ) {
            return i->second;
        }
        return CharsetTraits(Recognizer::CODES_UNKNOWN, charset, normalConverter);
    }
protected:
    typedef std::map<CodesEnum,CharsetTraits> CodeToTraits;
    typedef std::map<std::string,CharsetTraits> CharsetToTraits;

    void addItem(CodesEnum code, const std::string & charset) {
        addItem(code, charset, normalConverter);
    }

    void addItem(CodesEnum code, const std::string & charset,
                const TextConverter & convertor ) {
        addItem(code, charset, CharsetTraits(code, charset, convertor));
    }

    void addItem(CodesEnum code, const std::string & charset,
            const CharsetTraits & traits) {
        typedef CodeToTraits::value_type Item;
        code2Traits.insert(Item(code, traits));
        typedef CharsetToTraits::value_type RItem;
        charset2Traits.insert(RItem(charset, traits));
    }

    void addAlias(CodesEnum aliasCode, const std::string & aliasCharset,
            CodesEnum code) {
        const CodeToTraits::const_iterator it(code2Traits.find(code));
        if( it == code2Traits.end() ) {
            throw AddAliasForUninitializedCharset(aliasCharset);
        }
        addItem(aliasCode, aliasCharset, it->second);
    }
private:
    CodeToTraits code2Traits;
    CharsetToTraits charset2Traits;
    CharsetTraits defaultTraits;
    NormalTextConverter normalConverter;
    UtfTextConverter utfConverter;
    KoiKoiTextConverter koiKoiConverter;
    WinWinTextConverter winWinConverter;
};

#endif /* CHARSET_MAP_H_031020121518 */
