#ifndef CONTENT_TYPE_H
#define CONTENT_TYPE_H

#define _GNU_SOURCE 1
#include <map>
#include <vector>
#include <string>
#include <queue>
#include <magic.h>
#include <mail_getter/mime_type.h>

#include <boost/thread/mutex.hpp>
#include <boost/thread/tss.hpp>
#include <boost/smart_ptr/scoped_ptr.hpp>

typedef std::map<std::string, MimeType> MailGetterMimeMap;

typedef std::vector<std::string> ExtVector;

struct MimeTypeComparer {
    bool operator()(const MimeType& first, const MimeType& second) const {
        return first.type() == second.type()
            ? first.subtype() < second.subtype()
            : first.type() < second.type();
    }
};
typedef std::map<MimeType, ExtVector, MimeTypeComparer> LibmagicIssuesMap;

class MagicCookie {
public:
    MagicCookie(const std::string& magicFile);
    std::string getMime(const std::string& data);

private:
    class MagicWrapper {
    public:
        MagicWrapper(int flags);
        ~MagicWrapper();

        void load(const char* magic_file);
        const char* getMime(const void* buffer, size_t length);

    private:
        magic_t openCookie(int flags) const;

        magic_t magic_;
    };

    MagicWrapper magicWrapper_;
};

class MagicCookiesHolder {
public:
    MagicCookiesHolder(const std::string& magicFilePath);
    std::string getMime(const std::string& data) const;

private:
    std::string magicFilePath_;
    mutable boost::thread_specific_ptr<MagicCookie> cookiePtr_;
};

class IContentTypeDetector {
public:
    virtual ~IContentTypeDetector() {}
    virtual MimeType detectByFilename(const std::string& filename) const = 0;
    virtual MimeType detectByContent(const std::string& body) const = 0;
    virtual MimeType detect(const std::string& filename, const std::string& body = "") const = 0;
};

class ContentTypeDetector: public IContentTypeDetector
                         , private boost::noncopyable {
public:
    ContentTypeDetector(const std::string& mimeFilePath, const std::string& issuesFilePath,
                        const std::string& magicFilePath);

    MimeType detectByFilename(const std::string& filename) const;
    MimeType detectByContent(const std::string& body) const;
    MimeType detect(const std::string& filename, const std::string& body = "") const;

    /**
     * Modify mime type in case of:
     * message/rfc822 ( EML-8522 )
     * multipart/[any] (EML-9197)
     * We can't allow user to attach parts with those mime-types.
     */
    static void adjustType(MimeType& mType);

    static const int DEFAULT_MAGIC_FLAGS;

private:
    bool libmagicBadResult(const MimeType& mType, const std::string& filename) const;
    void loadMimeMap(const std::string& filename);
    void loadIssuesMap(const std::string& filename);

    MailGetterMimeMap mimeMap_;
    LibmagicIssuesMap issuesMap_;
    MagicCookiesHolder magicCookiesHolder_;
};

#endif
