#ifndef MIME_TYPE_H
#define MIME_TYPE_H

#include <string>
#include <map>
#include <stdexcept>

/**
 * Represents "mime type" - pair of type/subtype.
 * If we can't recognize type, it defaults will be set.
 * @todo integrate this class with MailCompose (use instead of std::pair)
 */

class MimeTypeInvalidType : public std::runtime_error {
public:
    explicit MimeTypeInvalidType(const std::string& what) :
        std::runtime_error(what) {}
};

class MimeTypeInvalidString : public std::runtime_error {
public:
    explicit MimeTypeInvalidString(const std::string& what ) :
        std::runtime_error(what) {}
};


class MimeType {
    public:
        typedef std::map<std::string, std::string> Params;
    public:
        MimeType();
        explicit MimeType(const std::string& mimeType);
        MimeType(const std::string& type, const std::string& subtype, const Params& params = Params());
        const std::string& type() const;
        const std::string& subtype() const;
        const std::string& param(const std::string& key) const;
        const Params& params() const;
        const std::string toString() const;
        const std::string typeAndSubtype() const {
            return type() + "/" + subtype();
        }
        void setMimeType(const std::string& type_ins, const std::string& subtype_ins, const Params& params);
        void setMimeType(const std::string& mimeType);
        void setDefaultMimeType();
        bool isDefaultMimeType() const;

        void setType(const std::string& type);
        void setSubtype(const std::string& subtype);
        void setParams(const Params& params);

        bool operator==(const MimeType& other) const;
        bool operator!=(const MimeType& other) const;
    private:
        static bool isValidMimeType(const std::string& type, const std::string& subtype);
        static void fixInvalidMimetype(std::string& type, std::string& subtype);
        static Params splitParams(const std::string& paramStr, const std::string& delimiters = ":;\r\n");
    public:
        /// default type, see http://tools.ietf.org/html/rfc2045#section-6.4
        static const std::string DEFAULT_TYPE;
        /// default subtype, see http://tools.ietf.org/html/rfc2045#section-6.4
        static const std::string DEFAULT_SUBTYPE;
        /// delimeter of type and subtype
        static const char DELIMITER='/';

        static const std::string EMPTY_PARAM_VALUE;

        static const char PARAMS_DELIMITER = ';';

        static const char PARAM_EQ = '=';
    private:
        std::string type_;
        std::string subtype_;
        Params params_;
};

std::ostream& operator<<(std::ostream& o, const MimeType& mimeType);

#endif
