#ifndef MAIL_GETTER_MULCAGATE_CLIENT_H
#define MAIL_GETTER_MULCAGATE_CLIENT_H

#include <functional>
#include <mail_getter/mulcagate/http.h>
#include <mail_getter/mulcagate/errors.h>
#include <mail_getter/logging.h>
#include <mail_getter/io.h>
#include <mail_getter/types.h>

namespace mail_getter {
namespace mulcagate {

struct Settings {
    std::string storageNameSpace;
    std::string service;
    std::string host;
    unsigned port = 0;
    unsigned connectTimeoutMs = 1000;
    unsigned getTimeoutMs = 1000;
    unsigned putTimeoutMs = 1000;
    unsigned retries = 0;
    bool keepAlive = false;
};

using SettingsPtr = std::shared_ptr<Settings>;

class MulcagateClient {
public:
    using AsyncHandler = Hook<std::string>;

    MulcagateClient(SettingsPtr settings, http::HttpClientPtr httpClient,
            logging::LogPtr logger, const tvm::Ticket& tvmTicket, const std::string& requestId)
            : settings_(std::move(settings)), httpClient_(std::move(httpClient)),
              logger_(std::move(logger)), tvmTicket_(tvmTicket), requestId_(requestId) {
    }

    template <typename Handler = io::sync_context>
    auto getWhole(const Stid& stid, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, AsyncHandler> init(handler);
        asyncGetWhole(stid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getByRange(const Stid& stid, const Range& range, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, AsyncHandler> init(handler);
        asyncGetByRange(stid, range, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getXml(const Stid& stid, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, AsyncHandler> init(handler);
        asyncGetXml(stid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto putData(const std::string& baseId, const std::string& data, std::chrono::seconds ttl,
            Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, AsyncHandler> init(handler);
        asyncPutData(baseId, data, ttl, init.handler);
        return init.result.get();
    }

private:
    void asyncGetWhole(const Stid& stid, AsyncHandler handler) const {
        HttpArguments args = {{}};
        return aget(stid, {{}}, std::move(args), std::move(handler));
    }

    void asyncGetByRange(const Stid& stid, const Range& range, AsyncHandler handler) const {
        http::Headers headers = {{ {"Range", {"bytes=" + toString(range)}} }};
        HttpArguments args = {{}};
        return aget(stid, std::move(headers), std::move(args), std::move(handler));
    }

    void asyncGetXml(const Stid& stid, AsyncHandler handler) const {
        HttpArguments args = {{ {"gettype", {"xml"}} }};
        return aget(stid, http::Headers(), std::move(args), std::move(handler));
    }

    void asyncPutData(const std::string& baseId, const std::string& data, std::chrono::seconds ttl,
            AsyncHandler handler) const {
        HttpArguments args = {{ {"elliptics", {"1"}},
                {"expire", {std::to_string(ttl.count()).append("s")}} }};
        return aput(baseId, data, std::move(args), std::move(handler));
    }

    void aget(const Stid& stid, http::Headers headers, HttpArguments args,
            AsyncHandler handler) const;
    void aput(const std::string& baseId, const std::string& data, HttpArguments args,
            AsyncHandler handler) const;

    SettingsPtr settings_;
    http::HttpClientPtr httpClient_;
    logging::LogPtr logger_;
    tvm::Ticket tvmTicket_;
    std::string requestId_;
};

typedef std::shared_ptr<const MulcagateClient> MulcagateClientPtr;

} // namespace mulcagate
} // namespace mail_getter

#endif // MAIL_GETTER_MULCAGATE_CLIENT_H
