#ifndef HASH_PROVIDER_H_152328032012
#define HASH_PROVIDER_H_152328032012

#include <string>
#include <vector>
#include <ctime>
#include <boost/algorithm/string/split.hpp>
#include <boost/assign/list_of.hpp>
#include <boost/algorithm/string.hpp>

#include <butil/digest.h>
#include <butil/split_key_value.h>
#include <mail_getter/base64_url.h>
#include <mail_getter/sms_sid_parser.h>
#include <mail_getter/vdirect/keys_storage.h>

namespace vdirect {

struct HashProvider {
    virtual ~HashProvider() {}

    virtual std::string hash(const std::string & url) const = 0;
};

typedef std::unique_ptr<HashProvider> HashProviderPtr;

class UidHashProvider : public HashProvider {
public:
    UidHashProvider(const KeysStorage & keys, const std::string & uid )
    : keys(keys), uid(uid) {
    }

    virtual std::string hash(const std::string & url) const {
        return keys.defaultKeyName() + ',' + calculateHash(url, keys.defaultKey());
    }

    bool valid(const std::string & url, const std::string & hashToCheck) const {
        const std::pair<std::string,std::string> res = splitPair(hashToCheck, ',');
        return calculateHash(url, keys.key(res.first)) == res.second;
    }
protected:
    std::string calculateHash(const std::string & url, const std::string & key) const {
        std::string retval = md5_raw(url + uid + key);
        encode_base64url(retval, true);
        return retval;
    }
private:
    const KeysStorage & keys;
    std::string uid;
};

typedef UidHashProvider UidHashValidator;

class SmsHashProvider : public HashProvider {
public:
    SmsHashProvider(const KeysStorage & keys, const std::string & smsHash )
    : keys(keys), smsHash(smsHash) {
    }

    virtual std::string hash(const std::string & url) const {
        return keys.defaultKeyName() + ',' + smsHash + ','
                + calculateHash(url, keys.defaultKey(), smsHash);
    }

    static std::string calculateHash(const std::string & url,
            const std::string & key,
            const std::string & smsHash) {
        std::string retval = md5_raw(url + smsHash + key);
        encode_base64url(retval, true);
        return retval;
    }
private:
    const KeysStorage & keys;
    std::string smsHash;
};

class SmsHashValidator {
public:
    virtual ~SmsHashValidator() {}
    SmsHashValidator(const KeysStorage & keys, std::time_t linkTtl )
    : keys(keys), linkTtl(linkTtl) {
    }

    bool valid(const std::string & url, const std::string & hash) const {
        std::vector<std::string> values;
        boost::split ( values, hash, boost::is_any_of ( "," ) );
        if(values.size() < 3) {
            throw std::out_of_range("VdirectSmsHashProvider::valid() invalid hash format: " + hash);
        }
        const std::string & keyName = values[0];
        const std::string & smsSid = values[1];
        const std::string & hashValue = values[2];
        if( validSmsSid(smsSid) ) {
            return calculateHash( url, keys.key(keyName), smsSid) == hashValue;
        }
        return false;
    }

    std::string calculateHash(const std::string & url, const std::string & key,
            const std::string & smsSid) const {
        return SmsHashProvider::calculateHash(url, key, smsSid);
    }
protected:
    virtual bool validSmsSid(const std::string & smsSid) const {
        const std::chrono::seconds ttl(linkTtl);
        const auto smsParser = mail_getter::sms::getParser(ttl);
        smsParser->parse(smsSid);
        return true;
    }
private:
    const KeysStorage & keys;
    std::time_t linkTtl;
};

} //namespace vdirect

#endif /* HASH_PROVIDER_H_152328032012 */
