#ifndef KEYS_STORAGE_H_145027032013
#define KEYS_STORAGE_H_145027032013

#include <map>
#include <string>
#include <stdexcept>
#include <istream>

namespace vdirect {

class KeysStorage {
public:
    static std::string delimiter() {
        return ":";
    }

    KeysStorage()
        : defaultKey_(keys.end())
    { }

    KeysStorage(std::istream& input)
        : defaultKey_(keys.end())
    {
        if (!input) {
            throw std::invalid_argument("Cannot open vdirect keys input for reading");
        }
        bool firstKey = true;
        for (std::string line; std::getline(input, line); ) {
            std::size_t pos = line.find_first_of(delimiter());

            if (pos == std::string::npos) {
                throw std::invalid_argument("Cannot find delimiter in the vdirect keys' file");
            }

            addKey(line.substr(0, pos), line.substr(pos+1), firstKey);

            firstKey = false;
        }
    }

    void addKey( const std::string & keyName, const std::string & key, bool asDefault = false ) {
        std::pair<Iter,bool> res = keys.insert(std::make_pair(keyName, key));
        if(res.second && asDefault) {
            defaultKey_ = res.first;
        }
    }
    void setDefaultKey(const std::string & keyName) {
        Iter i = keys.find(keyName);
        if( i==keys.end() ) {
            throw std::out_of_range("VdirectKeyStorage::setDefaultKey() no such key: " + keyName);
        }
        defaultKey_ = i;
    }
    const std::string key(const std::string & keyName) const {
        Iter i = keys.find(keyName);
        if( i==keys.end() ) {
            throw std::out_of_range("VdirectKeyStorage::key() no such key: " + keyName);
        }
        return i->second;
    }
    const std::string defaultKey() const {
        return getDefaultKeyEntry().second;
    }
    const std::string defaultKeyName() const {
        return getDefaultKeyEntry().first;
    }
    bool empty() const {
        return keys.empty();
    }
private:
    typedef std::map<std::string, std::string> KeysMap;
    typedef KeysMap::const_iterator Iter;
    Iter::reference getDefaultKeyEntry() const {
        if(defaultKey_ == keys.end()) {
            throw std::out_of_range("VdirectKeyStorage::getDefaultKeyEntry() no default key.");
        }
        return *defaultKey_;
    }
    KeysMap keys;
    Iter defaultKey_;
};

} // namespace vdirect

#endif /* KEYS_STORAGE_H_145027032013 */
