#include <boost/range/adaptor/transformed.hpp>
#include <boost/range/algorithm/transform.hpp>
#include <boost/algorithm/string/join.hpp>
#include <macs/mime_part_factory.h>
#include <mail_getter/MessageAccessDefinitions.h>

const std::string wholeMessageHid = "0";
const std::string rootHid = "1";

struct QuotesAppender {
    typedef std::string result_type;
    std::string operator()(const std::string& s) const {
        return "\"" + s + "\"";
    }
};

struct KeyValuePrinter {
    typedef std::string result_type;
    std::string operator()(const std::pair<std::string, std::string>& p) const {
        return quotesAppender_(p.first) + ": " + quotesAppender_(p.second);
    }
    QuotesAppender quotesAppender_;
};

std::ostream& operator<<(std::ostream& o, const MetaAttributes& attrs) {
    o << '{';
    o << boost::join(attrs | boost::adaptors::transformed(KeyValuePrinter()), ", ");
    o << '}';
    return o;
}

std::ostream& operator<<(std::ostream& o, const MetaLevel& level) {
    o << '[';
    o << boost::join(level | boost::adaptors::transformed(QuotesAppender()), ", ");
    o << ']';
    return o;
}

std::string getStidWithoutPrefix(const std::string& stid) {
    const static std::string prefix = "mulca:2:";
    return stid.substr(0, prefix.size()) == prefix
        ? stid.substr(prefix.size())
        : stid;
}

std::size_t parsePositiveValue(const MetaAttributes& ma, const std::string& name) {
    long long res = 0;
    auto it = ma.find(name);
    if (it != ma.end() && !it->second.empty()) {
        if (!boost::conversion::try_lexical_convert(it->second, res)) {
            std::ostringstream s;
            s << __PRETTY_FUNCTION__ << ": can't parse meta part, bad format of the " << name
                    << " attribute";
            throw std::runtime_error(s.str());
        }
        if (res < 0) {
            std::ostringstream s;
            s << __PRETTY_FUNCTION__ << ": can't parse meta part, " << name
                    << " attribute is negative";
            throw std::runtime_error(s.str());
        }
    }
    return static_cast<std::size_t>(res);
}

MetaPart metaAttributesToMetaPart(MetaAttributes ma) {
    auto offset = parsePositiveValue(ma, "offset");
    auto length = offset + parsePositiveValue(ma, "length");
    return macs::MimePartFactory().hid(std::move(ma["id"]))
            .contentType(std::move(ma["content_type.type"]))
            .contentSubtype(std::move(ma["content_type.subtype"]))
            .name(std::move(ma["content_type.name"]))
            .charset(std::move(ma["content_type.charset"]))
            .encoding(std::move(ma["content_transfer_encoding"]))
            .contentDisposition(std::move(ma["content_disposition.value"]))
            .fileName(std::move(ma["content_disposition.filename"]))
            .cid(std::move(ma["content_id"]))
            .offsetBegin(offset)
            .offsetEnd(length)
            .release();
}

MetaAttributes metaPartToMetaAttributes(const MetaPart& mp) {
    MetaAttributes ma;

    ma.insert({"id", mp.hid()});
    ma.insert({"content_type.type", mp.contentType()});
    ma.insert({"content_type.subtype", mp.contentSubtype()});
    ma.insert({"content_type.name", mp.name()});
    ma.insert({"content_type.charset", mp.charset()});
    ma.insert({"content_transfer_encoding", mp.encoding()});
    ma.insert({"content_disposition.value", mp.contentDisposition()});
    ma.insert({"content_disposition.filename", mp.fileName()});
    ma.insert({"content_id", mp.cid()});

    ma.insert({"offset", std::to_string(mp.offsetBegin())});
    ma.insert({"length", std::to_string(mp.length())});

    return ma;
}
