#include <internal/attachment_storage.h>
#include <internal/MessageStorageBlob.h>
#include <mail_getter/content_type.h>
#include <butil/StrUtils/StrUtils.h>
#include <stdexcept>
#include <boost/lexical_cast.hpp>
#include <macs/mime_part_factory.h>

using std::string;
using std::vector;

namespace mail_getter {

MessageStorageBlob::MessageStorageBlob(const string& stid, storage::ServicePtr storageService)
        : cached_(false), attStorage_(std::make_shared<AttachmentStorageImpl>(move(storageService), "")),
          st_id_(stid) {
}

void MessageStorageBlob::loadAttachments(OptYieldContext yc) {
    if (cached_) {
        return;
    }

    vector<string> ids;
    ids.push_back(st_id_);

    int result = attStorage_->get(ids, cachedAttachments_, yc);
    if (result != 0) {
        throw std::runtime_error("Can't get message from mulca: " + boost::lexical_cast<std::string>(result));
    }
    cached_ = true;
}

const AbstractAttachment& MessageStorageBlob::getCached() const {
    assert(cached_);
    return *(cachedAttachments_[0]);
}

OptMetaPart MessageStorageBlob::getHeaderStruct(const string& hid, OptYieldContext yc) {
    if (hid != rootHid) {
        return boost::none;
    }

    loadAttachments(yc);

    const auto& cached = getCached();
    MimeType mtype(cached.getContentType());
    return macs::MimePartFactory()
        .hid(hid)
        .contentType(mtype.type())
        .contentSubtype(mtype.subtype())
        .fileName(cached.getFilename())
        .offsetBegin(cached.getBegin())
        .offsetEnd(cached.getEnd())
        .release();
}

MetaLevel MessageStorageBlob::getBodyStruct(const string&, OptYieldContext) {
    throw std::logic_error("not implemented");
}

OptString MessageStorageBlob::getHeader(const string&, OptYieldContext) {
    throw std::logic_error("not implemented");
}

OptString MessageStorageBlob::getBody(const string& hid, OptYieldContext yc) {
    if (hid == rootHid) {
        loadAttachments(yc);

        return getCached().getBody();
    } else {
        return boost::none;
    }
}

string MessageStorageBlob::getWhole(OptYieldContext) {
    throw std::logic_error("not implemented");
}

MetaParts MessageStorageBlob::getMetaParts(OptYieldContext yc) {
    const MetaPart metaPart = getHeaderStruct(rootHid, yc).get();
    return {{metaPart.hid(), metaPart}};
}

} // namespace mail_getter
