#include <boost/algorithm/string/predicate.hpp>
#include <boost/range/algorithm/transform.hpp>

#include <butil/butil.h>

#include <macs/mime_part_factory.h>
#include <mimeparser/rfc2047.h>
#include <internal/MessageStorageRange.h>
#include <internal/MessageStorageString.h>
#include <internal/HeaderParser.h>
#include <mail_getter/UTFizer.h>
#include <internal/pa_log.h>

using namespace std;

namespace mail_getter {

MetaAttributes getMessageHeaderParsed(const std::string& hdr) {
    MetaAttributes res;
    HeaderParser hp(res, hdr);
    hp.parse(0, 0, hdr.size());
    return res;
}

bool hasXmlPart(const std::string& data) {
    static const std::string openMetaTag = "\n<message>";
    if (boost::starts_with(data, "<?xml")) {
        const auto endLinePos = data.find('\n');
        if (endLinePos != std::string::npos &&
                data.compare(endLinePos, openMetaTag.size(), openMetaTag) == 0) {
            return true;
        }
    }
    return false;
}

void eraseXmlPart(std::string& data) {
    static const std::string closeMetaTag = "</message>\n";
    if (hasXmlPart(data)) {
        auto endXmlPos = data.find(closeMetaTag);
        if (endXmlPos != std::string::npos) {
            endXmlPos += closeMetaTag.size();
            data.erase(0, endXmlPos);
        }
    }
}

MessageStorageRange::MessageStorageRange(const Stid& stid, storage::ServicePtr storageService,
        logging::LogPtr logger, MetaParts metaParts, const Recognizer::Wrapper& recognizer)
        : stid_(stid), storageService_(std::move(storageService)), logger_(std::move(logger)),
          metaParts_(std::move(metaParts)), inlineMessages_(getInlineMessages(metaParts_)),
          recognizer(&recognizer) {
    if (stid_.empty()) {
        throw std::invalid_argument("can't initialize mulca storage with empty st_id");
    }
    if (!storageService_) {
        std::ostringstream ss;
        ss << __PRETTY_FUNCTION__ << ", stid=" << stid_ << ": need mail storage service";
        throw std::runtime_error(ss.str());
    }
}

OptMetaPart MessageStorageRange::getHeaderStruct(const Hid& hid, OptYieldContext yc) {
    if (hid == wholeMessageHid) {
        auto hdr = getHeader(rootHid, yc);
        if (!hdr) {
            return boost::none;
        }
        auto messageHeadersParsed = getMessageHeaderParsed(*hdr);
        std::string subject = messageHeadersParsed["subject"];

        std::string charset = "utf-8";
        subject = mulca_mime::decode_rfc2047(subject, charset);
        UTFizer::process(*recognizer, charset, subject);

        if (subject.empty()) {
            subject = "No subject";
        }
        subject.append(".txt");

        return macs::MimePartFactory().hid(hid).contentType("message").contentSubtype("rfc822")
                .fileName(subject).release();
    }

    auto mpi = metaParts_.find(hid);
    return mpi != metaParts_.end() ? mpi->second : getHeaderStructInlineMessage(hid, yc);
}

MetaLevel MessageStorageRange::getBodyStruct(const Hid& hid, OptYieldContext yc) {
    MetaLevel res;
    int i = 1;
    for (; ; i++) {
        auto subhid = hid + "." + std::to_string(i);
        if (metaParts_.find(subhid) != metaParts_.end()) {
            res.push_back(std::move(subhid));
        } else {
            break;
        }
    }

    if (i == 1) {
        auto imi = inlineMessages_.find(hid);
        if (imi != inlineMessages_.end()) {
            res.push_back(hid + ".1");
        } else {
            auto inlineRes = getBodyStructInlineMessage(hid, yc);
            if (inlineRes) {
                res = std::move(*inlineRes);
            }
        }
    }

    return res;
}

OptString MessageStorageRange::getMessageHeader(OptYieldContext yc) {
    if (!messageHeader_) {
        auto mpi = metaParts_.find(rootHid);
        if (mpi == metaParts_.end()) {
            return boost::none;
        } else if (mpi->second.offsetBegin() == 0) {
            log("no headers in message");
            messageHeader_ = std::string();
        } else {
            messageHeader_ = getByRange({0, mpi->second.offsetBegin() - 1}, yc);
            eraseXmlPart(*messageHeader_);
        }
    }
    return messageHeader_;
}

OptString MessageStorageRange::getPartHeader(const MetaPart& mp, OptYieldContext yc) {
    const auto& data = getRawData(yc);
    auto endPos = mp.offsetBegin();

    if (!endPos) {
        return boost::none;
    }

    auto startPos = data.rfind("\n--", (endPos - 2));
    if (startPos == std::string::npos) {
        return boost::none;
    }
    startPos = data.find('\n', ++startPos);
    if (startPos == std::string::npos) {
        return boost::none;
    }
    ++startPos;

    if (startPos >= endPos) {
        return boost::none;
    }

    return data.substr(startPos, endPos - startPos);
}

OptString MessageStorageRange::getHeader(const Hid& hid, OptYieldContext yc) {
    if (hid == rootHid) {
        return getMessageHeader(yc);
    }

    auto headerInlineMessage = getHeaderInlineMessage(hid, yc);

    if (!headerInlineMessage) {
        auto mpi = metaParts_.find(hid);
        if (mpi == metaParts_.end()) {
            std::ostringstream ss;
            ss << "getHeader error: not find meta part, stid=" << stid_ << ", hid=" << hid;
            log(ss.str());
            return boost::none;
        }
        auto res = getPartHeader(mpi->second, yc);
        if (!res) {
            std::ostringstream ss;
            ss << "getHeader error: corrupted data, stid=" << stid_ << ", hid=" << hid;
            log(ss.str());
            return std::string();
        }
        return res;
    } else {
        return headerInlineMessage;
    }
}

std::string MessageStorageRange::getByRange(const Range& range, OptYieldContext yc) {
    const PaLog paLog(pa::mulca, "", "get_range", stid_);
    error_code ec;
    const auto data = applyOptYieldContext(yc, ec, [&] (auto token) {
        return storageService_->getByRange(stid_, range, token);
    });
    paLog();

    if (ec) {
        std::ostringstream s;
        s << "range message storage getByRange(stid=" << stid_ << ", range=" << toString(range)
                << ") returns error: " << ec.category().name() << ": " << ec.message();
        throw MessageStorageRangeError(s.str());
    }
    return data;
}

OptString MessageStorageRange::getBody(const Hid& hid, OptYieldContext yc) {
    if (hid == wholeMessageHid) {
        return getWhole(yc);
    }

    auto bodyInlineMessage = getBodyInlineMessage(hid, yc);

    if (!bodyInlineMessage) {
        auto mpi = metaParts_.find(hid);
        if (mpi != metaParts_.end()) {
            return mpi->second.length() == 0 ? std::string() :
                    getByRange({mpi->second.offsetBegin(), mpi->second.offsetEnd() - 1}, yc);
        } else {
            std::ostringstream ss;
            ss << "getBody error: not find meta part, stid=" << stid_ << ", hid=" << hid;
            log(ss.str());
            return boost::none;
        }
    } else {
        return bodyInlineMessage;
    }
}

std::string MessageStorageRange::getWhole(OptYieldContext yc) {
    auto whole = getRawData(yc);
    eraseXmlPart(whole);
    return whole;
}

const std::string& MessageStorageRange::getRawData(OptYieldContext yc) {
    if (!rawData_) {
        const PaLog paLog(pa::mulca, "", "range_get_blob", stid_);
        error_code ec;
        rawData_ = applyOptYieldContext(yc, ec, [&] (auto token) {
            return storageService_->getBlob(stid_, token);
        });
        paLog();
        if (ec) {
            std::ostringstream s;
            s << "range message storage getRawData(stid=" << stid_ << ") returns error: "
                    << ec.category().name() << ": " << ec.message();
            throw MessageStorageRangeError(s.str());
        }
    }

    return *rawData_;
}

MessageStorage *MessageStorageRange::findInlineMessage(const Hid& hid, OptYieldContext yc) {
    for (auto& im : inlineMessages_) {
        const auto& imHid = im.first;
        const auto hidPrefix = imHid + ".";
        if (hidPrefix.size() < hid.size() && !hid.compare(0, hidPrefix.size(), hidPrefix)) {
            auto& imStorage = im.second;
            if (!imStorage) {
                auto message = getInlineMessage(imHid, yc);
                if (!message) {
                    return nullptr;
                }
                imStorage = std::make_unique<MessageStorageString>(std::move(*message), hidPrefix);
            }
            return imStorage.get();
        }
    }
    return nullptr;
}

OptMetaPart MessageStorageRange::getHeaderStructInlineMessage(const Hid& hid, OptYieldContext yc) {
    auto inlmes = findInlineMessage(hid, yc);
    return !inlmes ? boost::none : inlmes->getHeaderStruct(hid, yc);
}

MessageStorageRange::OptMetaLevel MessageStorageRange::getBodyStructInlineMessage(const Hid& hid, OptYieldContext yc) {
    auto inlmes = findInlineMessage(hid, yc);
    return !inlmes ? boost::none : OptMetaLevel(inlmes->getBodyStruct(hid, yc));
}

OptString MessageStorageRange::getHeaderInlineMessage(const Hid& hid, OptYieldContext yc) {
    auto inlmes = findInlineMessage(hid, yc);
    return !inlmes ? boost::none : inlmes->getHeader(hid, yc);
}

OptString MessageStorageRange::getBodyInlineMessage(const Hid& hid, OptYieldContext yc) {
    auto inlmes = findInlineMessage(hid, yc);
    return !inlmes ? boost::none : inlmes->getBody(hid, yc);
}

OptString MessageStorageRange::getInlineMessage(const Hid& hid, OptYieldContext yc) {
    using namespace boost::algorithm;

    auto message = getBody(hid, yc);
    if (!message) {
        return message;
    }
    auto it = metaParts_.find(hid);
    if (it == metaParts_.end()) {
        return message;
    }
    const auto& transferEncoding = it->second.encoding();
    if (iequals(transferEncoding, "base64")) {
        *message = decode_base64(*message);
    } else if (iequals(transferEncoding, "quoted-printable")) {
        *message = decode_qp(*message);
    }
    return message;
}

MetaParts MessageStorageRange::getMetaParts(OptYieldContext) {
    return metaParts_;
}

} // namespace mail_getter
