#include <internal/MessageStorageString.h>
#include <internal/MetaParser.h>
#include <macs/mime_part_factory.h>

#include <sstream>
#include <string.h>
#include <algorithm>
#include <stdexcept>

#include <boost/lexical_cast.hpp>

namespace mail_getter {

const int MessageStorageString::inlineLevel{14};

MessageStorageString::MessageStorageString(std::string _content, const std::string &_prefix)
        : content(std::move(_content)), prefix(_prefix), parsed(false) {
}

MessageStorageString::~MessageStorageString() {
}

void MessageStorageString::parse() {
    if (!parsed) {
        mulca_mime::part parts(content, nullptr);
        parts.parse(inlineLevel, 0, content.size());
        splitToParts(parts, prefix + "1");
        parsed=true;
    }
}

void MessageStorageString::splitToParts(const mulca_mime::part &p, const std::string &id) {
    headersOffsets.insert({id, {p.header.offset(), p.header.length()}});

    macs::MimePartFactory mpf;
    mpf.hid(id).contentType(removeCRLF(p.header.content_type.type))
            .contentSubtype(removeCRLF(p.header.content_type.subtype))
            .name(removeCRLF(p.header.content_type.name))
            .charset(removeCRLF(p.header.content_type.charset))
            .encoding(removeCRLF(p.header.content_transfer_encoding))
            .contentDisposition(removeCRLF(p.header.content_disposition.value))
            .fileName(removeCRLF(p.header.content_disposition.filename))
            .cid(removeCRLF(p.header.content_id))
            .offsetBegin(p.offset())
            .offsetEnd(p.offset() + p.length());

    meta.emplace(id, mpf.release());
    int idAddon=1;
    for (mulca_mime::parts_t::const_iterator i=p.parts.begin(); i!=p.parts.end(); i++,idAddon++) {
        std::ostringstream wrk;
        wrk<<id<<"."<<idAddon;
        splitToParts(*i,wrk.str().c_str());
    }
}

OptMetaPart MessageStorageString::getHeaderStruct(const std::string &hid, OptYieldContext) {
    parse();

    MetaParts::iterator mpi;

    if ((mpi=meta.find(hid)) != meta.end()) {
        return mpi->second;
    }
    return boost::none;
}

MetaLevel MessageStorageString::getBodyStruct(const std::string &hid, OptYieldContext) {
    parse();

    MetaLevel res;
    int i;
    for (i=1;; i++) {
        std::ostringstream wrk;
        wrk<<hid<<"."<<i;
        std::string subhid(wrk.str().c_str());
        if(meta.find(subhid) != meta.end()) {
            res.insert(res.end(), subhid);
        } else {
            break;
        }
    }

    return res;
}

OptString MessageStorageString::getHeader(const std::string &hid, OptYieldContext) {
    parse();

    auto off = headersOffsets.find(hid);
    if (off != headersOffsets.end()) {
        return content.substr(off->second.begin, off->second.length);
    }
    return boost::none;
}

OptString MessageStorageString::getBody(const std::string &hid, OptYieldContext) {
    parse();

    auto mpi=meta.find(hid);
    if (mpi != meta.end()) {
        return content.substr(mpi->second.offsetBegin(), mpi->second.length());
    }
    return boost::none;
}

std::string MessageStorageString::getWhole(OptYieldContext) {
    return content;
}

MetaParts MessageStorageString::getMetaParts(OptYieldContext) {
    parse();
    return meta;
}

} // namespace mail_getter
