#include <internal/MetaParser.h>
#include <sstream>
#include <algorithm>
#include <mail_getter/UTFizer.h>
#include <butil/xml/routines.h>
#include <butil/StrUtils/Iconv.h>
#include <cstring>

MetaParser::MetaParser(const Recognizer::Wrapper& recognizer)
    : Expat("UTF-8"),
      start_handler(start),
      recognizer(&recognizer) {
    parserReset();
}

namespace {

MetaAttributes fillMetaAttributes(const char **attrs) {
    MetaAttributes ma;
    for (int i = 0; attrs[i]; i += 2) {
        if (attrs[i+1] == nullptr) {
            throw std::invalid_argument(__PRETTY_FUNCTION__ + std::string(
                    ": can't parse meta part, invalid attribute value"));
        }
        ma[attrs[i]] = removeCRLF(attrs[i+1]);
    }
    return ma;
}

}

void MetaParser::start(void *me, const char *el, const char **attrs) {
    if (strcmp(el, "part")) {
        return;
    }

    MetaParser *tclass = static_cast<MetaParser*>(me);
    auto mp = metaAttributesToMetaPart(fillMetaAttributes(attrs));

    const auto hid = mp.hid();
    if (!hid.empty()) {
        tclass->parts->emplace(hid, std::move(mp));
    }
}

void MetaParser::parserReset() {
    reset("UTF-8");
    setUserData(this);
    setStartElementHandler(start_handler);
}

int MetaParser::process(const std::string &source, MetaParts *mparts) {
    if (!mparts) {
        return METAPARSER_INVALIDPARAM;
    }
    if (source.empty()) {
        return METAPARSER_OK;
    }
    parts = mparts;
    std::string doc = source;
    const std::string encoding = butil::getXmlEncoding(source);
    UTFizer::process(*recognizer, encoding, doc);
    rc = parse(doc);
    if (rc != XML_STATUS_OK) {
        return rc = METAPARSER_XMLERROR;
    }
    return METAPARSER_OK;
}

std::string MetaParser::parserError() {
    std::string errorStr;
    if (rc > 0) {
        std::ostringstream oss;
        oss << Iconv::error(rc) << " at offset " << rc;
        errorStr.assign(oss.str().c_str());
    } else if (!rc) {
        errorStr.assign("no error");
    } else if (rc == METAPARSER_CANTRECODE) {
        errorStr.assign("linked iconv can't make cp1251 -> utf-8 recoding");
    } else if (rc == METAPARSER_UNKNOWN) {
        errorStr.assign("unknown iconv error");
    } else if (rc == METAPARSER_INVALIDPARAM) {
        errorStr.assign("invalid input parameters");
    } else if (rc == METAPARSER_XMLERROR) {
        errorStr.assign("xml parsing error: ");
        errorStr.append(error());
    } else {
        errorStr.assign("unknown error");
    }
    return errorStr;
}
