#include <mail_getter/alias_class_list.h>

#include <stdexcept>
#include <locale>

#include <boost/property_tree/ptree.hpp>
#include <boost/property_tree/xml_parser.hpp>

#include <boost/algorithm/string/case_conv.hpp>

#include <boost/bind.hpp>
#include <boost/range/algorithm/for_each.hpp>

#include <boost/optional/optional.hpp>

using namespace boost::property_tree;

using boost::algorithm::to_lower_copy;

namespace {

static const char configLocaleName[] = "en_US.UTF-8";

typedef void (AliasClassList::*ParseNode)(const ptree&);

void parseChilds(const ptree& node, const std::string& name, AliasClassList* aliasClassList,
    ParseNode parseNode)
{
    boost::range::for_each(node.equal_range(name),
        boost::bind(parseNode, aliasClassList, boost::bind(&ptree::value_type::second, _1))
    );
}

typedef void (AliasClassList::*ParseNodeArg)(const ptree&, const std::string&);

void parseChilds(const ptree& node, const std::string& name, AliasClassList* aliasClassList,
    ParseNodeArg parseNode, const std::string& arg)
{
    boost::range::for_each(node.equal_range(name),
        boost::bind(parseNode, aliasClassList, boost::bind(&ptree::value_type::second, _1), arg)
    );
}

bool hasFlag(const ptree& attr, const std::string& name) {
    return attr.get<std::string>(name, "n") == "y";
}

} // namespace

AliasClassList::AliasClassList(const std::string& config) {
    ptree root;
    read_xml(config, root, xml_parser::no_comments, std::locale(configLocaleName));

    const ptree& mimeTypesNode = root.get_child("mime-types");
    parseChilds(mimeTypesNode, "type", this, &AliasClassList::parseTypeNode);
}

bool AliasClassList::canBeThumbnailed(const MimeType& mType) const {
    return mimeTypesThumbSet_.count(mType.typeAndSubtype()) != 0;
}

bool AliasClassList::canBeRotated(const MimeType& mType) const {
    return mimeTypesRotatableSet_.count(mType.typeAndSubtype()) != 0;
}

bool AliasClassList::viewForbidden(const MimeType& mType) const {
    return mimeTypesNoViewSet_.count(mType.typeAndSubtype()) != 0;
}

bool AliasClassList::canBeBrowsed(const MimeType& mType) const {
    return mimeTypesBrowserSet_.count(mType.typeAndSubtype()) != 0;
}

const std::string& AliasClassList::defaultAlias() {
    static const std::string result = "general";
    return result;
}

const std::string& AliasClassList::getMimeAlias(const MimeType& mType) const {
    AliasMap::const_iterator it = mimeTypesAliasMap_.find(mType.typeAndSubtype());
    if (mimeTypesAliasMap_.end() != it) {
        return it->second;
    }
    it = mimeTypesAliasMap_.find( to_lower_copy( mType.type() + MimeType::DELIMITER ) );
    if (mimeTypesAliasMap_.end() != it) {
        return it->second;
    }
    return defaultAlias();
}

const std::string& AliasClassList::getExtAlias(const std::string& ext) const {
    AliasMap::const_iterator it = extAliasMap_.find( to_lower_copy( ext ) );
    if (extAliasMap_.end() != it) {
        return it->second;
    }
    return defaultAlias();
}

const std::string& AliasClassList::getAlias(const std::string& ext, const MimeType& mType) const {
    const std::string& result = getMimeAlias(mType);
    if (result != defaultAlias()) {
        return result;
    }

    return getExtAlias(ext);
}

void AliasClassList::parseTypeNode(const ptree& typeNode) {
    const std::string alias = typeNode.get<std::string>("<xmlattr>.class");

    parseChilds(typeNode, "mime", this, &AliasClassList::parseMimeNode, alias);
    parseChilds(typeNode, "ext", this, &AliasClassList::parseExtNode, alias);
}

void AliasClassList::parseMimeNode(const ptree& mimeNode, const std::string& alias) {
    const std::string mimeType = to_lower_copy(mimeNode.get_value<std::string>());
    const std::size_t delimPos = mimeType.find(MimeType::DELIMITER);
    if (delimPos == std::string::npos || delimPos == 0u) {
        throw std::runtime_error("Broken MimeType wildcard \"" + mimeType + "\" in class \"" + alias + "\"");
    }

    const std::pair<AliasMap::iterator, bool> addMimeTypeResult
            = mimeTypesAliasMap_.insert(std::make_pair(mimeType, alias));
    if (!addMimeTypeResult.second) {
        throw std::runtime_error("Dublicate class \"" + alias + "\" for MimeType wildcard \"" + mimeType + "\""
                                 + ", already set as \"" + addMimeTypeResult.first->second + "\"");
    }

    boost::optional<const ptree&> attr = mimeNode.get_child_optional("<xmlattr>");
    if (attr) {
        parseMimeAttr(attr.get(), mimeType);
    }
}

void AliasClassList::parseMimeAttr(const ptree& mimeAttr, const std::string& mimeType) {
    if (hasFlag(mimeAttr, "thumb"))
        mimeTypesThumbSet_.insert(mimeType);

    if (hasFlag(mimeAttr, "rotatable"))
        mimeTypesRotatableSet_.insert(mimeType);

    if (hasFlag(mimeAttr, "noview"))
        mimeTypesNoViewSet_.insert(mimeType);

    if (hasFlag(mimeAttr, "browser"))
        mimeTypesBrowserSet_.insert(mimeType);
}

void AliasClassList::parseExtNode(const ptree& extNode, const std::string& alias) {
    const std::string extension = to_lower_copy(extNode.get_value<std::string>());
    if (extension.empty()) {
        throw std::runtime_error("Empty extension in class \"" + alias + "\"");
    }

    const std::pair<AliasMap::iterator, bool> addExtResult
            = extAliasMap_.insert(std::make_pair(extension, alias));
    if (!addExtResult.second) {
        throw std::runtime_error("Dublicate class \"" + alias + "\" for extension \"" + extension + "\""
                                 + ", already set as \"" + addExtResult.first->second + "\"");
    }
}
