#include <mail_getter/attach_sid_keys.h>

#include <fstream>
#include <sstream>

#include <butil/butil.h>
#include <yamail/data/deserialization/yajl.h>

namespace mail_getter::attach_sid {

KeyContainer parseKeyContainer(const ptree& tree) {
    const auto aesKeysPath = tree.get<std::string>("aes_keys");
    const auto hmacKeysPath = tree.get<std::string>("hmac_keys");

    return parseKeyContainer(aesKeysPath, hmacKeysPath);
}

KeyContainer parseKeyContainer(const std::string& aesKeysPath, const std::string& hmacKeysPath) {
    const auto readKey = [] (const std::string& path) {
        try {
            std::ifstream file;
            file.exceptions(std::ifstream::failbit | std::ifstream::badbit);
            file.open(path);
            std::stringstream ss;
            ss << file.rdbuf();
            return ss.str();
        } catch (const std::exception& e) {
            throw std::runtime_error("Can't read key from file \"" + path + "\": " + e.what());
        }
    };

    const std::string aes = readKey(aesKeysPath);
    const std::string hmac = readKey(hmacKeysPath);

    std::map<std::string, crypto::AesKey> aesKeys;
    std::map<std::string, crypto::HmacKey> hmacKeys;

    const auto readKeys = [] (auto& keys, const std::string& json, const std::string& keyName) {
        const auto keysTmp = yamail::data::deserialization::fromJson<std::map<std::string, std::string>>(json);
        for (const auto& key : keysTmp) {
            keys.emplace(key.first, decode_base64(key.second));
        }
        if (keys.empty()) {
            throw std::runtime_error("no one " + keyName + " keys was read");
        }
    };

    readKeys(aesKeys, aes, "aes");
    readKeys(hmacKeys, hmac, "hmac");

    return {aesKeys, hmacKeys};
}

Keys initKeys(const ptree& tree, const KeyContainer& keyContainer) {
    const std::string aesKeyId = tree.get<std::string>("current_aes_key_id");
    const std::string hmacKeyId = tree.get<std::string>("current_hmac_key_id");

    Keys keys(keyContainer, aesKeyId, hmacKeyId);

    return keys;
}

} // namespace mail_getter::attach_sid