#include <internal/attachment_storage.h>
#include <mail_getter/SimpleAttachment.h>
#include <mail_getter/AttachShieldCrypto.h>
#include <mail_getter/MessageAccessDefinitions.h>
#include <internal/stid_parsers.h>

#include <butil/butil.h>

namespace mail_getter {

using std::string;

inline std::string encryptId(const part_id::Temporary& partId, std::chrono::seconds timeout, const attach_sid::Keys& keys) {
    const attach_sid::Packer packer(timeout, keys);
    return encode_url(packer(partId));
}

int AttachmentStorageImpl::add(const AbstractAttachment& attachment, std::string& id,
        MetaInfo& metaInformation, std::chrono::seconds expireTimeout, std::chrono::seconds shieldTimeout,
        const attach_sid::Keys& keys, OptYieldContext yc) {
    const int CANT_SAVE_FILE = -1;
    string headerAndBody = "";

    headerAndBody += encode_base64(attachment.getFilename());
    headerAndBody += '\n';
    headerAndBody += encode_base64(attachment.getContentType());
    headerAndBody += '\n';
    headerAndBody += '\n';
    headerAndBody += attachment.getBody();

    error_code ec;
    id = applyOptYieldContext(yc, ec, [&] (auto token) {
        return storage_->putBlob(tmpStorageServiceId, headerAndBody, expireTimeout, token);
    });
    if (ec) {
        return CANT_SAVE_FILE;
    }

    const part_id::Temporary partId {id, rootHid};
    const auto sid = encryptId(partId, shieldTimeout, keys);
    const auto prefix = webAttachHost_ + "/message_part_real/?sid=" + sid;
    metaInformation.previewUrl = prefix + "&thumb=y";
    metaInformation.viewLargeUrl = prefix + "&no_disposition=y";
    metaInformation.downloadUrl = prefix + "&name=" + encode_url(attachment.getFilename());
    metaInformation.sid = encryptId(partId, expireTimeout, keys);

    return 0;
}

int AttachmentStorageImpl::get(const std::vector<std::string>& ids, VectorOfAttachments& result, OptYieldContext yc) {
    const int BROKEN_FILE = -1;
    const int CANT_FIND_FILE = -2;

    for (std::vector<std::string>::const_iterator it = ids.begin();
            it != ids.end(); ++it) {
        error_code ec;
        const auto headerAndBody = applyOptYieldContext(yc, ec, [&] (auto token) {
            return storage_->getBlob(*it, token);
        });
        if (ec) {
            return CANT_FIND_FILE;
        }

        size_t firstEndl = headerAndBody.find('\n', 0);
        if (firstEndl == string::npos) {
            return BROKEN_FILE;
        }

        size_t secondEndl = headerAndBody.find('\n', firstEndl + 1);
        if (secondEndl == string::npos) {
            return BROKEN_FILE;
        }

        size_t thirdEndl = headerAndBody.find('\n', secondEndl + 1);
        if (thirdEndl == string::npos) {
            return BROKEN_FILE;
        }

        if ((firstEndl + 1 >= headerAndBody.size() ||
                (firstEndl + 1 > secondEndl))) {
            return BROKEN_FILE;
        }

        if (headerAndBody.begin() + static_cast<long>(thirdEndl) + 1 > headerAndBody.end()) {
            return BROKEN_FILE;
        }

        const string encodedFilename(headerAndBody, 0, firstEndl);
        const string encodedContentType(headerAndBody, firstEndl + 1, secondEndl - (firstEndl + 1));

        auto filename = decode_base64(encodedFilename);
        auto contentType = decode_base64(encodedContentType);
        auto body = std::string(headerAndBody.begin() + static_cast<long>(thirdEndl) + 1, headerAndBody.end());

        result.push_back(std::make_shared<SimpleAttachment>(std::move(filename),
                std::move(contentType), std::move(body), thirdEndl + 1, headerAndBody.size()));
    }
    return 0;
}

struct GetStid : public boost::static_visitor<std::string> {
    std::string operator ()(const part_id::Old& value) const {
        return value.stid;
    }

    std::string operator ()(const part_id::Temporary& value) const {
        return value.stid;
    }

    std::string operator ()(const part_id::SingleMessagePart&) const {
        throw std::logic_error("mail_getter::part_id::SingleMessagePart id doesn't contain stid");
    }

    std::string operator ()(const part_id::MultipleMessagePart&) const {
        throw std::logic_error("mail_getter::part_id::MultipleMessagePart id doesn't contain stid");
    }
};

std::string decryptId(const std::string& encryptedId, const attach_sid::KeyContainer& keyContainer) {
    attach_sid::Unpacker unpacker(keyContainer);
    const auto partId = unpacker(decode_url(encryptedId));
    return boost::apply_visitor(GetStid(), partId);
}

}
