#include <mail_getter/content_type.h>
#include <fstream>
#include <stdexcept>
#include <string>
#include <vector>

#include <boost/algorithm/string.hpp>
#include <boost/thread/mutex.hpp>
#include <boost/thread/lock_guard.hpp>

const int ContentTypeDetector::DEFAULT_MAGIC_FLAGS = MAGIC_MIME;

typedef std::vector<std::string> StringArray;
using boost::algorithm::to_lower_copy;
using boost::algorithm::trim_copy;

magic_t MagicCookie::MagicWrapper::openCookie(int flags) const
{
    magic_t cookie = magic_open( flags );
    if (cookie == nullptr) {
        throw std::runtime_error(
                "magic_open() failed. Can't allocate magic buffer");
    }
    return cookie;
}

MagicCookie::MagicWrapper::MagicWrapper(int flags)
    : magic_(openCookie(flags))
{}

MagicCookie::MagicWrapper::~MagicWrapper() {
    magic_close(magic_);
}

void MagicCookie::MagicWrapper::load(const char* magic_file)
{
    if ( magic_load( magic_, magic_file ) == -1 ) {
        throw std::runtime_error( "magic_load() failed. Error: "
                + std::string(magic_error( magic_ ) ) );
    }
}

const char* MagicCookie::MagicWrapper::getMime(const void* buffer, size_t length) {
    return magic_buffer( magic_, buffer, length );
}

MagicCookie::MagicCookie(const std::string& magicFile)
    : magicWrapper_(ContentTypeDetector::DEFAULT_MAGIC_FLAGS)
{
    magicWrapper_.load(magicFile.c_str());
}

std::string MagicCookie::getMime(const std::string& data)
{
    const char* res = magicWrapper_.getMime(data.data(), data.size());
    return res ? std::string(res) : std::string();
}

MagicCookiesHolder::MagicCookiesHolder(const std::string& magicFilePath)
    : magicFilePath_(magicFilePath)
{}

std::string MagicCookiesHolder::getMime(const std::string& data) const {
    MagicCookie* cookie = cookiePtr_.get();
    if (!cookie) {
        cookie = new MagicCookie(magicFilePath_);
        cookiePtr_.reset(cookie);
    }
    return cookie->getMime(data);
}

namespace{

std::string getExtension(const std::string& filename)
{
    std::string::size_type p = filename.rfind('.');
    if (p != filename.npos) {
            return to_lower_copy(trim_copy(filename.substr(p + 1) ) );
    }
    return "";
}

inline void splitLine(const std::string& s, StringArray& res, const std::string& sep = "\n\t\r ")
{
    boost::algorithm::split( res, s, boost::is_any_of(sep), boost::algorithm::token_compress_on );
}

}

bool ContentTypeDetector::libmagicBadResult(const MimeType& mType,
        const std::string& filename) const
{
    if ( filename.empty() ) {
        return false;
    }
    LibmagicIssuesMap::const_iterator imi = issuesMap_.find( mType );
    if ( imi == issuesMap_.end() ) {
        return false;
    }

    const std::string ext = getExtension( filename );
    const ExtVector& ev = imi->second;
    if ( ev.empty() ) {
        return true;
    }
    ExtVector::const_iterator evi = std::find( ev.begin(), ev.end(), ext );
    return evi != ev.end();
}

ContentTypeDetector::ContentTypeDetector(const std::string& mimeFilePath,
                                         const std::string& issuesFilePath,
                                         const std::string& magicFilePath)
    : magicCookiesHolder_(magicFilePath)
{
    loadMimeMap(mimeFilePath);
    loadIssuesMap(issuesFilePath);
}

void ContentTypeDetector::loadMimeMap(const std::string& filename)
{
    std::ifstream mf( filename.c_str() );
    if ( !mf ) {
        throw std::runtime_error( "Can't open mime.types file. Path " + filename );
    }

    std::string line;
    while( getline(mf,line) ) {
        const size_t found = line.find( '#' );
        if ( found != std::string::npos ) {
            line.erase( found );
        }

        StringArray strArray;
        splitLine( line, strArray );
        if ( strArray.size() < 2 ) {
            continue;
        }

        MimeType contentType;

        try {
            contentType.setMimeType(strArray[0]);
        } catch( const MimeTypeInvalidString& e ) {
            throw std::runtime_error( "Wrong MIME string in mime.types file. Error:\n\t"
                    + std::string(e.what() ) );
        }

        for( std::size_t i = 1; i < strArray.size(); ++i ) {
            mimeMap_.insert( MailGetterMimeMap::value_type(
                    to_lower_copy( strArray[i] ), contentType ) );
        }
    }
}

void ContentTypeDetector::loadIssuesMap(const std::string& filename)
{
    std::ifstream mf(filename.c_str());
    if (!mf) {
        throw std::runtime_error("Can't open libmagic.issues file. Path " + filename );
    }

    std::string line;
    while ( std::getline(mf, line) ) {
        const size_t found = line.find( '#' );
        if ( found != std::string::npos ) {
            line.erase( found );
        }

        StringArray strArray;
        splitLine( line, strArray );
        if ( strArray.empty() || strArray[0].empty() ) {
            continue;
        }
        MimeType mType( strArray[0] );
        ExtVector& ev = issuesMap_.insert(
                LibmagicIssuesMap::value_type( mType, ExtVector() ) ).first->second;
        for( std::size_t i = 1; i < strArray.size(); ++i ) {
            ev.push_back( to_lower_copy( strArray[i] ) );
        }
    }
}

        //Finds content type associated with filename's extention in mimeMap_
MimeType ContentTypeDetector::detectByFilename(const std::string& filename) const
{

    if (mimeMap_.empty()) {
        throw std::runtime_error("Empty mail_getter mime map.");
    }

    std::string ext = getExtension(filename);
    MimeType mType;
    MailGetterMimeMap::const_iterator it = mimeMap_.find(ext);
    if (it != mimeMap_.end()) {
        mType = it->second;
        adjustType(mType);
    }

    return mType;
}

        //Finds content type by file's body
MimeType ContentTypeDetector::detectByContent(const std::string& body) const
{
    if ( body.empty() ) {
        return MimeType();
    }
    const std::string mimeResult = magicCookiesHolder_.getMime(body);
    MimeType mType;
    if ( !mimeResult.empty() ) {
       mType.setMimeType( mimeResult );
    }
    return mType;
}

MimeType ContentTypeDetector::detect(const std::string& filename, const std::string& body) const
{
    MimeType result;
    if (!body.empty() ) {
        try {
            result = detectByContent(body);
        } catch (const MimeTypeInvalidString&) {
        }
    }
    if (result.isDefaultMimeType() || libmagicBadResult(result, filename) ) {
        result = detectByFilename(filename);
    }
    return result;
}

        //Modify mType if it is "message/rfc822" or "multipart/..." for some reason
void ContentTypeDetector::adjustType(MimeType& mType)
{
    if (("message" == mType.type() && "rfc822" == mType.subtype()) || ( "multipart" == mType.type())) {
        mType.setDefaultMimeType();
    }
}
