#include <mail_getter/http_response_parser.h>

#include <boost/variant/get.hpp>

#include <boost/range/empty.hpp>
#include <boost/range/algorithm_ext/erase.hpp>

#include <boost/algorithm/string.hpp>

namespace mail_getter { namespace http {

DataPart::DataPart() {
}

DataPart::DataPart( const std::string& rawPart ) {
    using namespace boost;

    const iterator_range<std::string::const_iterator>
            crlf = find_first( rawPart, "\r\n\r\n" ),
            contentTypePrefix = find_first( rawPart, "Content-Type: " );

    const std::string rawContentType( boost::end( contentTypePrefix ), boost::begin( crlf ) );
    if( boost::empty(rawContentType) ) {
        throw InvalidHttpResponse( "Can't parse http response part=[" + rawPart + "]" );
    }

    const std::string rawData( boost::end( crlf ), boost::end( rawPart ) );

    parseContentType( rawContentType );
    parseData( rawData );
}

DataPart::DataPart( const std::string& data, const std::string& contentType ) {
    parseContentType( contentType );
    parseData( data );
}

const MimeType& DataPart::getContentType() const {
    return contentType;
}

void DataPart::setContentType(const MimeType& mime) {
    contentType = mime;
}

const DataPart::DataType& DataPart::getData() const {
    return data;
}

void DataPart::setData(const DataType& data_) {
    data = data_;
}

const PlainData& DataPart::getPlainData() const {
    try {
        return boost::get<PlainData>( data );
    } catch( const boost::bad_get& e ) {
        throw IsNotPlainData( e.what() );
    }
}

bool DataPart::isPlain() const {
    return boost::get<PlainData>( &data ) != 0;
}

const MultipartData& DataPart::getMultipartData() const {
    try {
        return boost::get<MultipartData>( data );
    } catch( const boost::bad_get& e ) {
        throw IsNotMultipartData( e.what() );
    }
}

void DataPart::parseContentType( const std::string& rawContentType ) {
    contentType.setMimeType( rawContentType );
}

void DataPart::parseData( const std::string& rawData ) {
    if( contentType.type() == "multipart"
            && contentType.param( "boundary" ) != MimeType::EMPTY_PARAM_VALUE ) {
        data = MultipartData( rawData, contentType.param( "boundary" ) );
    } else {
        data = PlainData( rawData );
    }
}

bool operator==(const DataPart& lhs, const DataPart& rhs ) {
    return lhs.getContentType() == rhs.getContentType()
            && lhs.getData() == rhs.getData();
}

MultipartData::MultipartData( const std::string& rawData, const std::string& boundary ) {
    using namespace boost;

    const std::string crlf("\r\n");
    const std::string delimiter("\r\n--" + boundary);
    for( split_iterator<std::string::const_iterator> it =
            make_split_iterator( rawData, first_finder( delimiter, is_equal() ) );
            !it.eof();
            ++it ) {
        if( !empty( *it ) && starts_with( *it, crlf ) ) {
            parts.push_back( DataPart( erase_first_copy( copy_range<std::string>( *it ), crlf ) ) );
        }
    }
}

bool operator==( const MultipartData& lhs, const MultipartData& rhs ) {
    return lhs.parts == rhs.parts;
}

}

http::DataPart parseHttpResponse( const std::string& data, const std::string& contentType ) {
    return http::DataPart( data, contentType );
}

}
