#include <boost/range/algorithm/transform.hpp>
#include <macs/mime_part_factory.h>

#include <internal/meta_parts_converter_impl.h>
#include <internal/MetaParser.h>
#include <internal/MessageStorageString.h>
#include <internal/pa_log.h>

namespace mail_getter {

MetaParts addOffset(const MetaParts& metaParts, size_t offset) {
    MetaParts res;
    boost::transform(metaParts, std::inserter(res, res.end()), [offset] (const auto& v) {
        const auto& mp = v.second;
        auto mpNew = macs::MimePartFactory(mp).offsetBegin(mp.offsetBegin() + offset)
            .offsetEnd(mp.offsetEnd() + offset).release();
        return std::make_pair(mp.hid(), std::move(mpNew));
    });
    return res;
}

MetaParts MetaPartsConverterImpl::getMetaPartsFromXml(const Stid& stid, OptYieldContext yc) const {
    const PaLog paLog(pa::mulca, "", "range_get_xml", stid);
    error_code ec;
    const auto xml = applyOptYieldContext(yc, ec, [&] (auto token) {
        return storageService_->getXml(stid, token);
    });
    paLog();
    if (ec) {
        std::ostringstream s;
        s << __PRETTY_FUNCTION__ << ": " << "can't get meta for " << stid << ": "
                << ec.category().name() << ": " << ec.message();
        throw std::runtime_error(s.str());
    }

    MetaParser mp{*recognizer};
    MetaParts metaParts;
    if (mp.process(xml, &metaParts)) {
        std::ostringstream s;
        s << __PRETTY_FUNCTION__ << ": " << "can't parse meta for " << stid << ": " << mp.parserError();
        throw std::runtime_error( s.str() );
    }

    return addOffset(metaParts, xml.size());
}

MetaParts MetaPartsConverterImpl::extractMetaPartsFromInline(MessageAccessPtr messageAccess,
        const MetaPart& inlineMessagePart, OptYieldContext yc) const {
    const std::string part = messageAccess->getBody(inlineMessagePart.hid());
    const std::string hidPrefix = inlineMessagePart.hid() + ".";
    MessageStorageString storage(part, hidPrefix);
    const MetaParts inlineParts = storage.getMetaParts(yc);
    return addOffset(inlineParts, inlineMessagePart.offsetBegin());
}

}
