#include <butil/StrUtils/StrUtils.h>
#include <mail_getter/mime_type.h>
#include <string>
#include <sstream>
#include <stdexcept>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/algorithm/string/classification.hpp>

const std::string MimeType::DEFAULT_TYPE="application";
const std::string MimeType::DEFAULT_SUBTYPE="octet-stream";
const std::string MimeType::EMPTY_PARAM_VALUE="";

MimeType::MimeType()
{
    setDefaultMimeType();
}

MimeType::MimeType(const std::string& mimeType)
{
    setMimeType(mimeType);
}

MimeType::MimeType(const std::string& type, const std::string& subtype, const Params& params)
{
    setMimeType(type, subtype, params);
}

const std::string& MimeType::type() const
{
    return type_;
}

const std::string& MimeType::subtype() const
{
    return subtype_;
}

const std::string& MimeType::param(const std::string& key) const
{
    Params::const_iterator iter = params_.find(key);
    return iter == params_.end() ? EMPTY_PARAM_VALUE : iter->second;
}

const MimeType::Params& MimeType::params() const {
    return params_;
}

const std::string MimeType::toString() const
{
    std::stringstream ss;
    ss << type_ << DELIMITER << subtype_;
    for (Params::const_iterator iter = params_.begin(); iter != params_.end(); ++iter) {
        ss << PARAMS_DELIMITER << ' ' << iter->first << PARAM_EQ << iter->second;
    }
    return ss.str();
}

bool MimeType::operator==(const MimeType& other) const {
    return type_ == other.type_
        && subtype_ == other.subtype_
        && params_ == other.params_;
}

bool MimeType::operator!=(const MimeType& other) const {
    return !operator==(other);
}

void MimeType::setMimeType(const std::string& type_ins, const std::string& subtype_ins, const Params& params)
{
    using TStrUtils::stringToLower;
    using TStrUtils::trim;


    std::string type = trim(stringToLower(type_ins) );
    std::string subtype = trim(stringToLower(subtype_ins) );
    fixInvalidMimetype(type, subtype);

    if (isValidMimeType(type, subtype) ) {
        type_ = type;
        subtype_ = subtype;
        params_ = params;
    } else {
        throw MimeTypeInvalidType("MimeType::setMimeType(): Invalid MIME type - " + type + '/' + subtype);
    }
}

class KeyValueSplitter {
public:
    KeyValueSplitter(MimeType::Params& params)
        : params_(params)
    {}

    void operator()(const std::string& param) {
        std::vector<std::string> keyValue;
        boost::split(keyValue, param, boost::is_any_of("="));

        if (keyValue.size() != 2) {
            return;
        }

        boost::algorithm::trim(keyValue[0]);

        if (keyValue[0].empty()) {
            return;
        }

        boost::algorithm::trim(keyValue[1]);
        params_[keyValue[0]] = keyValue[1];
    }
private:
    MimeType::Params& params_;
};

MimeType::Params MimeType::splitParams(const std::string& paramStr, const std::string& delimiters)
{
    Params res;
    std::vector<std::string> parts;
    boost::split(parts, paramStr, boost::is_any_of(delimiters), boost::token_compress_on);
    std::for_each(parts.begin(), parts.end(), KeyValueSplitter(res));
    return res;
}

void MimeType::setMimeType(const std::string& mimeString)
{
    size_t delimiterPos = mimeString.find(DELIMITER);
    if (std::string::npos == delimiterPos) {
        throw MimeTypeInvalidString("MimeType::setMimeType(): Invalid mime string - " + mimeString);
    }
    size_t paramDelimPos = mimeString.find(PARAMS_DELIMITER);
    Params params;
    if (paramDelimPos != std::string::npos) {
        params = splitParams(mimeString.substr(paramDelimPos + 1));
    }
    std::string type = mimeString.substr(0, delimiterPos);
    std::string subtype = mimeString.substr(delimiterPos + 1, paramDelimPos - delimiterPos - 1);
    setMimeType(type, subtype, params);
}

void MimeType::setDefaultMimeType()
{
    type_ = DEFAULT_TYPE;
    subtype_ = DEFAULT_SUBTYPE;
}

bool MimeType::isDefaultMimeType() const
{
    return (type_ == DEFAULT_TYPE) && (subtype_ == DEFAULT_SUBTYPE);
}

bool MimeType::isValidMimeType(const std::string& type, const std::string& subtype)
{
    if (type.empty() ||
        std::string::npos != type.find_first_of(" /") ||
        std::string::npos != subtype.find_first_of(" /")) {
        return false;
    }
    return true;
}

void MimeType::fixInvalidMimetype(std::string& type, std::string& subtype)
{
    static const char* BAD_SYMBOLS = "\"\r\n\\ ;/";
    size_t np = type.find_last_of( BAD_SYMBOLS ) + 1;
    if (np != type.npos) {
        type.erase( 0, np );
    }
    np = subtype.find_first_of( BAD_SYMBOLS );
    if (np != subtype.npos) {
        subtype.erase( np );
    }
}

void MimeType::setType(const std::string& type) {
    type_ = type;
}

void MimeType::setSubtype(const std::string& subtype) {
    subtype_ = subtype;
}

void MimeType::setParams(const Params& params) {
    params_ = params;
}

std::ostream& operator<<(std::ostream& o, const MimeType& mimeType) {
    return o << mimeType.toString();
}
