#include <mail_getter/mulcagate/mulcagate_client.h>
#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace mail_getter {
namespace mulcagate {

using Milliseconds = std::chrono::milliseconds;

inline bool isHttpCodeSuccess(unsigned code) {
    return code / 100 == 2;
}

inline bool isHttpCodeServerError(unsigned code) {
    return code / 100 == 5;
}

struct RequestContext {
    ~RequestContext() = default;

    RequestContext(SettingsPtr settings, http::HttpClientPtr httpClient, logging::LogPtr logger,
            http::BaseParams params, MulcagateClient::AsyncHandler handler)
            : settings(std::move(settings)), httpClient(std::move(httpClient)),
              logger(std::move(logger)), params(std::move(params)), handler(std::move(handler)) {}

    SettingsPtr settings;
    http::HttpClientPtr httpClient;
    logging::LogPtr logger;
    http::BaseParams params;
    MulcagateClient::AsyncHandler handler;

    unsigned retry = 0;
};

using GetContext = RequestContext;

struct PostContext : public RequestContext {
    PostContext(SettingsPtr settings, http::HttpClientPtr httpClient, logging::LogPtr logger,
            http::BaseParams params, std::string data, MulcagateClient::AsyncHandler handler)
            : RequestContext(std::move(settings), std::move(httpClient), std::move(logger),
                    std::move(params), std::move(handler)),
              data(std::move(data)) {
    }

    std::string data;
};

struct GetRequest {
    void operator() (const GetContext& ctx, http::ResponseHandler handler) const {
        ctx.httpClient->aget(ctx.params, std::move(handler));
    }
};

struct PostRequest {
    void operator() (const PostContext& ctx, http::ResponseHandler handler) const {
        ctx.httpClient->apost(ctx.params, ctx.data, std::move(handler));
    }
};

template <typename Context, typename Request>
struct RequestExecuter : boost::asio::coroutine {
    using ContextPtr = std::shared_ptr<Context>;

    ContextPtr ctx;
    Request req;

    RequestExecuter(ContextPtr ctx, Request req) : ctx(std::move(ctx)), req(std::move(req)) {}

    void logRetry(const std::string &reason) const {
        ctx->logger->notice("mulcagate", "retrying request to mulcagate due to error: " + reason);
    }

    void operator() (error_code error, http::Response response) {
        try {
            reenter(*this) {
                while (ctx->retry++ < ctx->settings->retries) {
                    yield req(*ctx, *this);

                    if (error) {
                        logRetry(error.message());
                    } else if (isHttpCodeSuccess(response.code)) {
                        ctx->handler(error, response.body);
                        return;
                    } else if (isHttpCodeServerError(response.code)) {
                        error = error_code(Errors::internal);
                    } else if (response.code == 404) {
                        error = error_code(Errors::dataNotFound, "Mulcagate service responded with 404");
                        break;
                    } else if (response.code == 416) {
                        error = error_code(Errors::badRange, "Mulcagate service responded with 416");
                        break;
                    } else {
                        error = error_code(
                            Errors::httpCode,
                            "Mulcagate service responded with error code " + std::to_string(response.code)
                        );
                        break;
                    }
                }
                ctx->handler(error, std::string());
            }
        } catch (const std::exception& e) {
            error = error_code(Errors::exception, e.what());
            ctx->handler(error, std::string());
        }
    }
};

void MulcagateClient::aget(const Stid& stid, http::Headers headers, HttpArguments args,
        AsyncHandler handler) const {
    const auto urlPath = "/gate/get/" + encode_url(stid);
    args.add("service", settings_->service) ;
    http::BaseParams params = {settings_->host, settings_->port, urlPath,
            Milliseconds(settings_->connectTimeoutMs), Milliseconds(settings_->getTimeoutMs),
            settings_->keepAlive, headers, args, tvmTicket_, requestId_};
    auto ctx = std::make_shared<GetContext>(settings_, httpClient_, logger_,
            std::move(params), std::move(handler));
    auto request = RequestExecuter<GetContext, GetRequest>(std::move(ctx), GetRequest{});
    request(error_code(), http::Response());
}

void MulcagateClient::aput(const std::string& baseId, const std::string& data, HttpArguments args,
        AsyncHandler handler) const {
    const auto urlPath = "/gate/put/" + encode_url(baseId);
    args.add("service", settings_->service);
    args.add("ns", settings_->storageNameSpace);
    http::BaseParams params = {settings_->host, settings_->port, urlPath,
            Milliseconds(settings_->connectTimeoutMs), Milliseconds(settings_->putTimeoutMs),
            settings_->keepAlive, {{}}, args, tvmTicket_, requestId_};
    auto ctx = std::make_shared<PostContext>(settings_, httpClient_, logger_,
            std::move(params), data, std::move(handler));
    auto request = RequestExecuter<PostContext, PostRequest>(std::move(ctx), PostRequest{});
    request(error_code(), http::Response());
}

} // namespace mulcagate
} // namespace mail_getter
