#include <mail_getter/recognizerWrapper.h>
#include <internal/utf16.h>
#include <webmail-recognizer.h>

namespace Recognizer
{

static CodesEnum tryRecognizeUnknown(const std::string& text) {
    if (isValidUtf16(text)) {
        return CODES_UTF16;
    }
    return CODES_UNSUPPORTED;
}

class WrapperImpl : public Wrapper {
public:
    WrapperImpl() : recognizer() {}
    WrapperImpl(const char* languageDict, const char* languageWeights,
                const char* encodingDict)
        : recognizer(languageDict, languageWeights, encodingDict) {}

    TextTraits recognize(const std::string& text) const override {
        int code = 0;
        int lang = 0;
        const Result res(recognizer.AnalyzeText(text.c_str(), static_cast<unsigned int>(text.length()), code, lang));
        return res.succeeded() ? TextTraits(code, lang) : TextTraits();
    }

    CodesEnum recognizeEncoding(const std::string& text) const override {
        int code = 0;
        const Result res(recognizer.AnalyzeEncoding(text.c_str(), static_cast<unsigned int>(text.length()), code));
        if (res.succeeded()) {
            const CodesEnum res = convertEnum(code, CODES_UNSUPPORTED, CODES_MAX);
            return (res == CODES_UNKNOWN) ? tryRecognizeUnknown(text) : res;
        } else {
            return CODES_UNSUPPORTED;
        }
    }

    std::string isoNameByLanguageCode(LangsEnum lang) const override {
        return recognizer.IsoNameByLanguageCode(lang);
    }
private:
    TWebmailRecognizer recognizer;
};

WrapperPtr create() {
    return std::make_unique<WrapperImpl>();
}

WrapperPtr create(const char* languageDict, const char* languageWeights,
                  const char* encodingDict) {
    return std::make_unique<WrapperImpl>(languageDict, languageWeights, encodingDict);
}

}

