#include <mail_getter/sanitizer/sanitizer_response_processor.h>
#include <mail_getter/http_response_parser.h>

#include <boost/bind.hpp>
#include <boost/range/algorithm/find_if.hpp>

namespace mail_getter {

SanitizerParsedResponse SanitizerResponseProcessor::operator() (const std::string& data,
        const std::string& contentType) const {
    SanitizerParsedResponse parsedResponse;
    const http::DataPart dataPart = parseHttpResponse(data, contentType);
    parsedResponse.html = getHtmlPart(dataPart);
    parsedResponse.markup = getSanitizerMarkup(dataPart);
    return parsedResponse;
}

const std::string& SanitizerResponseProcessor::getHtmlPart(
        const http::DataPart& parsedData) const {
    const MimeType mimeToFind( "text/html; charset=utf-8" );

    const http::MultipartData& multipartData = parsedData.getMultipartData();
    http::MultipartData::DataParts::const_iterator html = boost::find_if(multipartData.parts,
            boost::bind( &http::DataPart::getContentType, _1) == mimeToFind);
    if (html == multipartData.parts.end()) {
        throw SanitizerParseException( "Html response from sanitizer not found" );
    }
    return html->getPlainData();
}

SanitizerMarkup SanitizerResponseProcessor::getSanitizerMarkup(
        const http::DataPart& parsedData) const {
    const MimeType mimeToFind("application/json; charset=utf-8");

    const http::MultipartData& multipartData = parsedData.getMultipartData();
    http::MultipartData::DataParts::const_iterator markupJson = boost::find_if(multipartData.parts,
            boost::bind(&http::DataPart::getContentType, _1) == mimeToFind);
    if (markupJson == multipartData.parts.end()) {
        throw SanitizerParseException( "Markup response from sanitizer not found" );
    }
    try {
        return parseSanitizerMarkup(markupJson->getPlainData());
    } catch (const std::exception &e) {
        throw SanitizerParseException( std::string("Cant parse markup response: ") + e.what() );
    }
}

}
