#include <macs/mime_part_factory.h>

#include <internal/service.h>
#include <internal/MessageStorageRange.h>
#include <internal/MessageStorageBlob.h>
#include <internal/attachment_storage.h>
#include <internal/stid_parsers.h>
#include <internal/meta_parts_converter_impl.h>
#include <internal/message_access_impl.h>
#include <internal/MessageAccessWithWindat.h>


namespace mail_getter {

bool ServiceImpl::needBlobStorage(const Stid& stid) const {
    const auto serviceId = mail_getter::extractServiceId(stid);
    if (!serviceId) {
        throw std::invalid_argument("Can't create message storage, bad stid=" + stid);
    }
    return serviceId == mail_getter::tmpStorageServiceId;
}

bool ServiceImpl::useMimes(const MetaParts& metaParts) const {
    return !metaParts.empty();
}

MessageStoragePtr ServiceImpl::createMessageStorage(MessageAccessParams params,
        const Recognizer::Wrapper& recognizer, OptYieldContext yc) const {
    if (needBlobStorage(params.stid)) {
        return std::make_unique<MessageStorageBlob>(params.stid, storageService);
    } else {
        auto metaParts = useMimes(params.parts) ? std::move(params.parts) :
                createMetaPartsConverter(recognizer)->getMetaPartsFromXml(params.stid, yc);
        return std::make_unique<MessageStorageRange>(params.stid, storageService, logger,
                std::move(metaParts), recognizer);
    }
}

MessageAccessPtr ServiceImpl::createMessageAccess(MessageAccessParams params,
        const Recognizer::Wrapper& recognizer, OptYieldContext yc) const {
    const auto stid = getStidWithoutPrefix(params.stid);
    params.stid = stid;

    return std::make_shared<MessageAccessImpl>(stid, createMessageStorage(std::move(params), recognizer, yc), yc);
}

MessageAccessPtr ServiceImpl::createMessageAccess(MessageAccessWithWindatParams params,
        const Recognizer::Wrapper& recognizer, OptYieldContext yc) const {
    auto mainMessageAccess = createMessageAccess(std::move(params.mainParams), recognizer, yc);

    MessageAccessByHid windatMessageAccesses;
    for (auto &hidWithMAParams : params.windatParams) {
        windatMessageAccesses.insert({std::move(hidWithMAParams.first),
            createMessageAccess(std::move(hidWithMAParams.second), recognizer, yc)} );
    }
    return std::make_shared<MessageAccessWithWindat>(std::move(mainMessageAccess),
            std::move(windatMessageAccesses));
}

AttachmentStoragePtr ServiceImpl::createAttachmentStorage(const std::string& webAttachHost) const {
    return std::make_shared<AttachmentStorageImpl>(storageService, webAttachHost);
}

MetaPartsConverterPtr ServiceImpl::createMetaPartsConverter(const Recognizer::Wrapper& recognizer) const {
    return std::make_shared<MetaPartsConverterImpl>(storageService, recognizer);
}

}  // namespace mail_getter
