#include <mail_getter/sms_sid_parser.h>

namespace mail_getter {
namespace sms {

static void checkSidFormat(const std::string& sid) {
    std::string::size_type pos = sid.find(',');
    if (pos == std::string::npos) {
        throw std::runtime_error("Broken sms-sid - no comma-separator found: " + sid);
    }
    const std::string prefix = sid.substr(0, pos);
    if (prefix != "v2") {
        throw std::runtime_error("unknown sms-sid format, prefix=" + prefix);
    }
}

class ParserImpl: public Parser {
public:
    ParserImpl(std::chrono::seconds timeToLive)
        : timeToLive_(timeToLive)
    {}

    TimedUidMid parse(const std::string& sid) const override {
        const std::string decrypted = decrypt(sid);
        checkSidFormat(decrypted);
        return parseTimedUidMid(decrypted);
    }

    std::string serialize(const TimedUidMid& timedUidMid) const override {
        std::string sid = "v2,";
        sid.append(yamail::data::serialization::toJson(timedUidMid).str());
        return encrypt(sid);
    }

private:
    TimedUidMid parseTimedUidMid(const std::string& sid) const {
        const std::string::size_type pos = sid.find(',');
        BOOST_ASSERT(pos > 0);
        const std::string serialized = sid.substr(pos+1);
        const TimedUidMid res = yamail::data::deserialization::fromJson<TimedUidMid>(serialized);
        checkTimestamp(res.ts);
        return res;
    }

    void checkTimestamp(time_t timestamp) const {
        time_t now = time(nullptr);
        if (now - timeToLive_.count() > timestamp) {
            throw std::runtime_error("Timestamp has expired " + boost::lexical_cast<std::string>(timestamp));
        }
    }

    std::string encrypt(const std::string& plainsid) const {
        try {
            auto encsid = crypto::encrypt_string(plainsid, KEY, IV);
            encode_base64url(encsid, true);
            return encsid;
        } catch (const std::exception& e) {
            throw std::runtime_error(std::string("Broken sid - encryption error ") + e.what());
        }
    }

    std::string decrypt(const std::string& sid) const {
        try {
            return crypto::decrypt_string(decode_base64url(sid), KEY, IV);
        } catch (const std::exception& e) {
            throw std::runtime_error(std::string("Broken sid - decryption error ") + e.what());
        }
    }

    const std::string KEY = "KeyKeyKeyKeyKey!";
    const std::string IV = "Anything";

    std::chrono::seconds timeToLive_;
};

ParserPtr getParser(std::chrono::seconds linkTTL) {
    return std::make_unique<ParserImpl>(linkTTL);
}

} // namespace sms
} // namespace mail_getter
