
#include <stdint.h>
#include <internal/utf16.h>

bool hasBom(const unsigned char* buf) {
    return (buf[0] == 0xfe && buf[1] == 0xff)
        || (buf[0] == 0xff && buf[1] == 0xfe);
}

bool isHighSurrogate(unsigned char first, unsigned char second) {
    uint16_t x = static_cast<uint16_t>((first << 8) + second);
    return 0xd800 <= x && x <= 0xdbff;
}

bool isLowSurrogate(unsigned char first, unsigned char second) {
    uint16_t x = static_cast<uint16_t>((first << 8) + second);
    return 0xdc00 <= x && x <= 0xdfff;
}

bool hasUnpairedSurrogates(const unsigned char* buf, size_t len) {
    for (size_t i = 0; i < len; i += 2) {
        if ( isHighSurrogate(buf[i], buf[i+1])) {
            if ( (i += 2) == len || !isLowSurrogate(buf[i], buf[i+1]) ) {
                return true;
            }
        } else if (isLowSurrogate(buf[i], buf[i+1])) {
            return true;
        }
    }
    return false;
}

bool isValidUtf16(const std::string& text) {
    if (text.size() < 2 || text.size() % 2 != 0) {
        return false;
    }

    const unsigned char* buf = reinterpret_cast<const unsigned char*>(text.c_str());

    if (!hasBom(buf)) {
        return false;
    }

    if (hasUnpairedSurrogates(buf + 2, text.size() - 2)) {
        return false;
    }

    return true;

}
