#include <vector>
#include <butil/xml/entities.h>
#include <butil/network/idn.h>
#include <mail_getter/vdirect/href_iterator.h>
#include <mail_getter/vdirect/secure_vdirect.h>

namespace {

const std::string redirectorPrefix = "http://mail.yandex.ru/r?url=";

} // namespace

const std::string SecureVdirect::linkParamPrefix = "&amp;l=";


std::string SecureVdirect::processA(const std::string& str) const {
    HrefIterator i(str), last;

    if( i==last ) {
        return str;
    }

    std::string res;
    HrefIterator::size_type textChunkBegin(0);
    for(; i!=last; textChunkBegin = i.last, ++i ) {
        res+= str.substr(textChunkBegin, i.first - textChunkBegin) + process(*i);
    }
    res += str.substr(textChunkBegin);
    return res;
}

void SecureVdirect::unwrap(std::string& str) const {
    const std::string pattern = "href=\"" + getUnwrapPattern();
    for(HrefIterator i(str, pattern), last; i!=last; ++i ) {
        const std::string::size_type pos = i.first - pattern.length();
        const std::string unwrapped( "href=\"" + unwrapUrl(*i));
        str.replace(pos, i.last - pos, unwrapped);
        i.last = pos + unwrapped.length();
    }
}

std::string SecureVdirect::encode(const http::url & url) {
    std::string punycoded;
    try {
        punycoded = idna::encode(url.server());
    }
    catch (const std::exception&) {
        punycoded = url.server();
    }
    if (url.portSpecified()) {
        punycoded += ":"
                + boost::lexical_cast<std::string>(url.port());
    }
    std::string retval = url.protocol() + "://";
    if( !url.login().empty() ) {
        retval += url.login();
        if(!url.password().empty()) {
            retval+= ":" + url.password();
        }
        retval += '@';
    }
    std::string uri = url.uri();
    retval += punycoded + decodeXmlEntities(uri);
    encode_base64url(retval, true);
    return retval;
}

std::string SecureVdirect::hash(const std::string & url) const {
    return hashProvider->hash(url);
}

std::string SecureVdirect::process(const std::string & urlString) const {
    const http::url url(urlString);
    if( url.protocol().empty() || boost::algorithm::starts_with(urlString, redirectorPrefix) ) {
        return urlString;
    }

    const std::string encodedUrl = encode(url);

    return getUnwrapPattern() + "?h=" + hash(encodedUrl)
            + linkParamPrefix + encodedUrl;
}

std::string SecureVdirect::unwrapUrl(const std::string & urlPart) const {
    const std::string originalUrl = urlPart.substr(
            urlPart.find(linkParamPrefix)+linkParamPrefix.length());
    return decode_base64url(originalUrl);
}

std::string SecureVdirect::makeUnwrapPattern() const {
    std::string retval = https() ? "https://" : "http://";
    retval += "mail";
    if(domain()[0] != '.' ) {
        retval += '.';
    }
    retval += domain() + '/' + scriptName();
    return retval;
}

std::string SecureVdirect::getUnwrapPattern() const {
    return unwrapPattern;
}
