/**
 * Виртуальная клавиатура
 */

(function($, Lego) {

    Lego.block['b-keyboard'] = function(params) {
        var $this = this, _this = this.get(0), rowsLayout = [14, 15, 14, 13];

        var $body = $(document.body);

        if (!params.lang) params.lang = '';

        if (params.fake !== false) {
            // Сейчас $this - временный элемент b-keyboard, удаляем его
            $this.remove();
            // Инициализируем настоящий элемент b-keyboard
            Lego.blockInit($body, '.b-keyboard');
        }

        // Блокируем выделение текста внутри блока
        _this.onmousedown = function() { return false; };
        _this.onselectstart = function() { return false; };

        var KEY_BLOCK = 'b-keyboard__key',
            KEY_PRESSED = KEY_BLOCK + '_pressed',
            KEY_SUPPRESSED = KEY_BLOCK + '_suppressed',
            KEY_MARKED = KEY_BLOCK + '_marked',
            KEY_DISABLED = KEY_BLOCK + '_disabled';

        var targetField = null;

        // Автоповтор нажатой клавиши
        var hold = false, holdInterval = null, holdDelay = 0;

        var unescapeHTMLRe = /(&(lt|gt|quot|apos|amp|#\d+);|.)/gi,
            unescapeHTMLHash = {lt: '<', gt: '>', quot: '"', apos: "'", amp: '&'};
        var unescapeHTMLReplacer = function(s, c, d) {
            return unescapeHTMLHash[d] || (d ? String.fromCharCode(d.substring(1)) : c);
        };

        // Установка языка и лэйаута
        var setters = {
            language: function(lang) {
                $this.data('lang', Lego.block['b-keyboard'].lang[lang]);
                if (targetField) targetField.focus();
            },

            modality: function(modality, capslock) {
                var accent = modality.indexOf('accent_') > -1 && modality.split("_")[1];

                // Если запрос на смену акцента - меняем акцент, оставляем текущую раскладку
                if (accent) {
                    accent = $this.data('accent') == accent ? 'normal' : accent;
                    modality = $this.data('modality') || 'normal';
                    capslock = $this.data('capslock') || false;
                // Если запрос на смену раскладки - меняем раскладку, оставляем текущий акцент
                } else {
                    accent = $this.data('accent') || 'normal';
                    modality = $this.data('modality') == modality ? 'normal' : modality;
                    capslock = capslock || false;
                }

                // Устанавливаем вычисленные значения раскладки и акцента и генерируем разметку
                $this
                    .data('langModality', $this.data('lang')[modality])
                    .data('modality', modality)
                    .data('capslock', capslock)
                    .data('accent', accent)
                    .html(generators.rows());

                Lego.blockInit($this, '.b-dropdown');
                actions.markLayout(modality, accent);
                actions.attachEvents();
                if (targetField) targetField.focus();
            }
        };

        // Генераторы разметки
        var generators = {
            rows: function() {
                var rows = '', start = 0;

                for (var i = 0; i < rowsLayout.length; i++) {
                    var rowLength = parseInt(rowsLayout[i]);

                    rows += '<table class="b-keyboard__row"><tr>' + generators.keys(start, rowLength) + '</tr></table>';

                    start += rowLength;
                }

                rows += generators.bottomRow();

                return rows;
            },

            keys: function(start, rowLength) {
                var keys = '';

                for (var key = start; key < start + rowLength; key++) {
                    keys += generators.key(key);
                }

                return keys;
            },

            key: function(key) {
                // Special keys can be defined either per-modality or per-language
                var specialKeys = $this.data('langModality').specialKeys || $this.data('lang').specialKeys;
                // Special keys are enumerated from 1
                var specialKey = specialKeys[key + 1];

                var keyClass = (specialKey ? (' ' + KEY_BLOCK + '_special-' + specialKey.type) : '');
                var keyLabel = (specialKey ? specialKey.label : $this.data('langModality').keys.charAt(key));

                // Certain layouts have accent modifiers
                var accent = $this.data('accent');
                var accentKeys = (accent && $this.data('langModality').accents) ? $this.data('langModality').accents[accent] : null;

                // Mark the accent modifier keys
                if (specialKey && specialKey.type.indexOf('accent') > -1) keyClass += ' ' + KEY_MARKED;

                if (accentKeys && !specialKey) {
                    if (accentKeys[key + 1]) {
                        // Set the alternative label (from accentKeys)
                        keyLabel = accentKeys[key + 1].label;
                    } else {
                        // Disable the button, if it's not of 'accent' type
                        keyClass += ' ' + KEY_DISABLED;
                    }
                }

                if (keyLabel == "shift" || keyLabel == "capslock" || keyLabel == "enter") {
                    keyLabel = '<img class="b-keyboard__key__image" src="' + Lego.params['lego-static-host'] + '/common/block/b-keyboard/key/b-keyboard__key.' + keyLabel + '.png" alt="' + keyLabel +'"/>';
                }

                if (keyLabel == '\u0000') keyLabel = '';

                return '<td class="b-keyboard__row__cell">' +
                        '<span class="' + KEY_BLOCK + keyClass + '">' +
                            '<span class="' + KEY_BLOCK + '-l"></span>' +
                            '<span class="' + KEY_BLOCK + '-m">' + keyLabel + '</span>' +
                            '<span class="' + KEY_BLOCK + '-r"></span>' +
                        '</span>' +
                       '</td>';
            },

            bottomRow: function() {
                return '<table class="b-keyboard__row">' +
                        '<tr>' +
                            '<td class="b-keyboard__row__cell">' +
                                generators.langSelector() +
                            '</td>' +
                            '<td class="b-keyboard__row__cell">' +
                                '<span class="b-keyboard__key b-keyboard__key_special-space">' +
                                    '<span class="b-keyboard__key-l"></span>' +
                                    '<span class="b-keyboard__key-m">\u0020</span>' +
                                    '<span class="b-keyboard__key-r"></span>' +
                                '</span>' +
                            '</td>' +
                            '<td class="b-keyboard__row__cell">' +
                                '<span class="b-keyboard__key b-keyboard__key-spacer"></span>' +
                            '</td>' +
                            '<td class="b-keyboard__row__cell">' +
                                '<span class="b-keyboard__key b-keyboard__key_special-alt b-keyboard__key_marked">' +
                                    '<span class="b-keyboard__key-l"></span>' +
                                    '<span class="b-keyboard__key-m">« » { } ~</span>' +
                                    '<span class="b-keyboard__key-r"></span>' +
                                '</span>' +
                            '</td>' +
                        '</tr>' +
                       '</table>';
            },

            langSelector: function() {
                return '<div class="b-keyboard__lang">' +
                        '<div class="b-keyboard__lang-i">' +
                            '<div class="b-dropdown b-dropdown_black g-js" onclick="return {name:\'b-dropdown\', direction: \'up\'}">' +
                                '<ul class="b-dropdown__list">'+
                                    generators.langItems() +
                                '</ul>' +
                            '</div>' +
                        '</div>' +
                       '</div>';
            },

            langItems: function() {
                // Sorted alphabetically (Russian)
                var langs = ['en', 'by', 'es', 'it', 'kz', 'de', 'ru', 'ua', 'fr'],
                    curLang = $this.data('lang').id;

                var items = '';

                for (var i = 0; i < langs.length; i++) {
                    var lang = langs[i],
                        langLabel = Lego.block['b-keyboard'].lang[lang].label;

                    // Build the language list, skipping the current one
                    if (lang != curLang) {
                        items += '<li class="b-dropdown__item">' +
                                    '<a class="b-dropdown__link b-keyboard__lang-' + lang + '" href="#">' +
                                        '<i class="b-keyboard__lang-ic"></i>' + langLabel +
                                    '</a>' +
                                  '</li>';
                    }
                }

                // Append current language to the bottom of the list
                items += '<li class="b-dropdown__item b-dropdown__visible">' +
                            '<a class="b-dropdown__or b-keyboard__lang-' + curLang + '" href="#">' +
                                '<i class="b-keyboard__lang-ic"></i>' + Lego.block['b-keyboard'].lang[curLang].label + '<i class="b-dropdown__or-ie">▼</i>' +
                            '</a>' +
                          '</li>';

                return items;
            }
        };

        // Обработчики нажатий на клавиши виртуальной клавиатуры
        var handlers = {
            key: function(key) {
                var specialKey = key.className.match(/b-keyboard__key_special-(\w+)/);

                if (specialKey) {
                    handlers.specialKey(key, specialKey[1]);
                } else {
                    handlers.normalKey(key);
                }

                targetField && $(targetField).keydown();
            },

            normalKey: function(key, type) {
                if (key && targetField) {
                    var $key = $(key);
                    if ($key.hasClass(KEY_DISABLED)) return;

                    // Получаем набранный символ
                    var keyValue = $key.find('.' + KEY_BLOCK + '-m').html().replace(unescapeHTMLRe, unescapeHTMLReplacer);

                    // Устанавливаем значение с учетом положения каретки
                    var $targetField = $(targetField);

                    if (type != 'backspace') {
                        if (type == 'enter') {
                            // если нажали экранный Enter в текстовом поле - сабмитим форму
                            if (targetField.tagName.toLowerCase() == 'input') {
                                try { $(targetField.form).submit(); } catch(e) {}
                                return;
                            }
                            else keyValue = '\n';
                        }
                        if (type == 'space') keyValue = ' ';

                        $targetField.insertAtCaretPos(keyValue);
                    } else {
                        $targetField.deleteAtCaretPos();
                    }
                }

                // Reset modality if we're in Shift mode
                if ($this.data('modality') == 'shift' && $this.data('capslock') === false) {
                    setters.modality('normal');
                }

                // Reset accent if we're in accent mode
                if ($this.data('accent').toString() != 'normal') {
                    setters.modality('accent_normal');
                }
            },

            specialKey: function(key, type) {
                switch (type) {
                    case 'lshift':
                    case 'rshift':
                        setters.modality('shift');
                        break;
                    case 'capslock':
                        setters.modality('shift', true);
                        break;
                    case 'alt':
                        setters.modality('alt');
                        break;
                    case 'accent_circumflex':
                    case 'accent_umlaut':
                    case 'accent_grave':
                    case 'accent_acute':
                    case 'accent_mod':
                        setters.modality(type);
                        break;
                    case 'backspace':
                    case 'atmark':
                    case 'enter':
                    default:
                        handlers.normalKey(key, type);
                        break;
                }
            }
        };

        // Различные действия
        var actions = {
            markLayout: function(type, accent) {
                // Mark modality button
                if (type == 'alt') {
                    $('.' + KEY_BLOCK + '_special-alt').addClass(KEY_SUPPRESSED);
                }

                if (type == 'shift') {
                    if ($this.data('capslock') === true)
                        $('.' + KEY_BLOCK + '_special-capslock').addClass(KEY_SUPPRESSED);
                    else {
                        $('.' + KEY_BLOCK + '_special-lshift').addClass(KEY_SUPPRESSED);
                        $('.' + KEY_BLOCK + '_special-rshift').addClass(KEY_SUPPRESSED);
                    }
                }

                // Mark accent button
                if (accent) {
                    $('.' + KEY_BLOCK + '_special-accent_' + accent).addClass(KEY_SUPPRESSED);
                }
            },

            switchLayout: function(layout) {
                var lang = layout.className.match(/b-keyboard__lang-(\w+)/);

                if (lang) {
                    setters.language(lang[1]);
                    $this.data('accent', 'normal');
                    setters.modality('normal');
                }
            },

            attachEventsOnce: function() {
                // Привязываем обработчики событий к клавишам виртуальной клавиатуры
                $this
                    .find('.' + KEY_BLOCK)
                    .live('click', function() {
                        targetField && $(targetField).keypress();
                        return false;
                    })
                    .live('mousedown', function(e) {
                        // Only allow left-button click
                        if (e.which && e.which != 1 || e.button && e.button != 1)
                            return false;

                        if (!$(this).hasClass(KEY_DISABLED)) {
                            $(this).addClass(KEY_PRESSED).data("pressed", 1);
                        }

                        // Первое нажатие
                        handlers.key(this);

                        // Включаем автоповтор
                        hold = true;

                        var self = this;
                        holdInterval = window.setInterval(function() {
                            if (hold === true) {
                                // Первые 3 цикла - пропускаем (задержка), потом начинаем повторять нажатую клавишу
                                if (holdDelay < 3) {
                                    holdDelay++;
                                } else {
                                    handlers.key(self);
                                }
                            }
                        }, 100);
                    })
                    .live('mouseup', function() {
                        // Выключаем автоповтор
                        clearHold();

                        $(this).removeClass(KEY_PRESSED).data("pressed", 0);

                        targetField && $(targetField).keyup();
                    });

                // Защита от кражи фокуса из активного поля ввода при промахивании мимо клавиши
                $this.click(function(e) {
                    if (targetField && !$(e.target).is('.' + KEY_BLOCK)) {
                        $(targetField).setCaretPos();
                    }
                });

                // Привязываем клавиатуру к первому полю ввода
                $('input, textarea').each(function() {
                    if ((this.tagName.toLowerCase() == 'input' && (this.type == 'text' || this.type == 'search')) ||
                        this.tagName.toLowerCase() == 'textarea') {
                        try {
                            this.focus();
                            $(this).saveCaretPos();
                            targetField = this;
                            return false;
                        } catch(e) {}
                    }
                });
            },

            attachEvents: function() {
                // вводить можно только в текстовые поля
                // можно написать input[type=text], но иногда не пишут type
                var $inputs = $('input').filter(function() {
                    return this.type === 'text' || this.type === 'search';
                }).add($('textarea'));

                // Привязываем клавиатуру к полю с активным фокусом
                $inputs
                    // mouseup нужен для ловли выделения
                    .bind('mouseup focus', function(e) {
                        // блокировка обработки фокуса после возвращения коретки на нужное место после ввода
                        if (blockFocus && e.type == 'focus') {
                            blockFocus = false;
                            return;
                        }
                        targetField = this;
                        $(this).saveCaretPos();
                    });

                var blockFocus = false;
                // IE - сохраняем позицию каретки в активном поле при событиях keypress и change
                if ($.browser.msie) {

                    var IESetCaretTimeout;

                    $inputs.bind('keypress change', function(e) {
                        // Таймаут обязателен, так как фокус нужно перевыставить после blur
                        if (IESetCaretTimeout) {
                            window.clearTimeout(IESetCaretTimeout);
                        }
                        blockFocus = true;
                        IESetCaretTimeout = window.setTimeout(function() {
                            $(targetField).setCaretPos();
                            IESetCaretTimeout = null;
                        }, 10);
                    });
                }

                // Инициализируем переключатель языков
                $this
                    .find('.b-dropdown__link')
                    .bind('click', function() { actions.switchLayout(this); return false; });

                // Защита от залипания подсветки нажатой клавиши
                $this
                    .find('.' + KEY_BLOCK)
                    .bind('mouseleave', function() {
                        var $this = $(this);
                        if ($this.data("pressed") == 1) $this.removeClass(KEY_PRESSED);

                        // Выключаем автоповтор
                        clearHold();
                    });
            }
        };

        if (params.fake === false) {
            var defaultLanguage = window.location.hostname.match(/.(ru|ua|by|kz)$/);

            actions.attachEventsOnce();

            $this.data('accent', 'normal');
            setters.language(params.lang || defaultLanguage && defaultLanguage[1] || 'en');
            setters.modality('normal');
        }

        /* Вспомогательные функции */
        function clearHold() {
            hold = false;
            holdDelay = 0;
            clearInterval(holdInterval);
        }
    };

    Lego.block['b-keyboard'].lang = {};

})(jQuery, window.Lego);

include('jquery.fn.caret.js');
include('lang/b-keyboard__lang_en.js');
include('lang/b-keyboard__lang_de.js');
include('lang/b-keyboard__lang_fr.js');
include('lang/b-keyboard__lang_it.js');
include('lang/b-keyboard__lang_es.js');
include('lang/b-keyboard__lang_ua.js');
include('lang/b-keyboard__lang_ru.js');
include('lang/b-keyboard__lang_by.js');
include('lang/b-keyboard__lang_kz.js');
