# -*- coding: utf-8 -*-
from blackbox import BlackboxResponseError

from mlcore.ml.models import MailList
from mlcore.subscribe import subscription_type as sub_type
from mlcore.tasks import base
from mlcore.utils.blackbox_util import get_uid_by_email

import logging
logger = logging.getLogger(__name__)


def get_maillist_by_email(email):
    """
    Смотрим, есть ли рассылка в базе ML по у email-у.
    В yandex-team-backend не смотрим намеренно.
    """
    maillists = MailList.objects.filter(email=email)
    if maillists:
        return maillists[0]
    return None


def exist_in_passport(email):
    # TODO: Нужно ли тут проверять imap_email?
    # Пример: есть пользователь vi@yandex-team.ru в паспорте и
    # пришел запрос на создание рассылки vi@yandex-team.com
    # Проверяя рассылку в паспорте, найдем: vi@yandex-team.com - пользовательский email.
    # Но в ML все рассылке вне домена yandex-team.ru будут созданы как vi-at-yandex-team-com@yandex-team.ru
    return bool(get_uid_by_email(email))


def handling_service_crm(context, email, email_destination):
    """
    Если системный алиас (отрс, сепаратор, crm) существует в паспорте, то нужно понять, чем он является в
    паспорте? Рассылкой или пользователем?
    Если в базе ML алиас найден, значит уже существует как рассылка и нужно подписать соотв. email_destination.
    Чтобы не делать асинхронную операцию подписки в ML по несколько раз - проверяем существование этой подписки в
    EmailSubscribers.
    Если в базе ML алиас не найден, значит в паспорте уже существует пользователь с таким email-ом и
    мы его со своей стороны в ML не создаем.

    Для tcrm@ мы НИЧЕГО НЕ ОТПИСЫВАЕМ в крон скрипте, а только подписываем.
    """

    """
    Не понял почему не отписываем. Люди жалуются.

    Набросал скрипт отписки для tcrmlite, может пригодится:

    from mlcore.ml.models import EmailSubscriber, MailList
    import requests
    r = requests.get('http://crm.yandex-team.ru/tcrmlite.txt')
    r.raise_for_status()
    emails = r.content.split()
    EMAIL = 'tcrm-lite@yandex-team.ru'
    maillists_should = MailList.objects.filter(email__in=emails).values_list('id', flat=True)
    subs = EmailSubscriber.objects.filter(email=EMAIL).values_list('list_id', flat=True)
    to_unsubscribe = set(subs) - set(maillists_should)
    from maillists.mlcore.tasks.low_level import unsubscribe_ml
    from mlcore.subscribe import subscription_type as sub_type
    for ml in MailList.objects.filter(id__in=to_unsubscribe):
        #t = unsubscribe_ml.delay(context={'initiator': 'lavrinenko'}, user=EMAIL, maillist=ml, type=sub_type.INBOX)
        #print t
        EmailSubscriber.objects.filter(list=ml, email=EMAIL).delete()
        ml.set_as_modified()
    """
    try:
        if email and exist_in_passport(email):
            if MailList.objects.filter(email=email):
                logging.info("subscribe tcrm@yandex-team.ru in %s ", email)
                m = MailList.objects.filter(email=email)[0]
                if email_destination not in m.emailsubscriber_set.all().values_list('email', flat=True):
                    base.subscribe_ml.delay(context, email_destination, m, sub_type.INBOX)
            else:
                logging.warning("WARNING: %s exist as user", email)
        else:
            destinations = [email_destination, ]
            base.create_alias_and_subscribe_users.delay(context, email, destinations, [])
    except BlackboxResponseError as exc:
        if exc.content and 'DB_EXCEPTION' in exc.message:
            logging.debug('Blackbox occurred db error with email: %s', email)
        else:
            raise


def handling_service_otrs(email, otrs_email):
    otrs_mail = 'otrs@otrs-mail.yandex-team.ru'
    otrs_inbox = 'otrs@otrs-inbox.yandex-team.ru'

    def _get_otrs_subscriber(maillist):
        otrs_subscribers = []
        if maillist is None:
            return []
        if maillist.emailsubscriber_set.filter(email=otrs_inbox):
            otrs_subscribers.append(otrs_inbox)
        if maillist.emailsubscriber_set.filter(email=otrs_mail):
            otrs_subscribers.append(otrs_mail)
        return otrs_subscribers

    maillist = get_maillist_by_email(email)
    context = {
        'initiator': 'robot_otrs_emails',
        'comment': 'Change or create redirect: %s -> %s' % (email, otrs_email),
    }
    exist_destination = _get_otrs_subscriber(maillist)
    if maillist and len(exist_destination) == 1 and exist_destination[0] == otrs_email:
        return
    elif maillist:
        # смотрим на существующего подписчика, если это otrs_inbox, то отписываем его и подписываем otrs_mail,
        # и наоборот
        if otrs_inbox in exist_destination and otrs_email == otrs_mail:
            base.subscribe_ml.delay(context, otrs_mail, maillist, sub_type.INBOX)
            base.unsubscribe_ml.delay(context, otrs_inbox, maillist, sub_type.INBOX)
        elif otrs_mail in exist_destination and otrs_email == otrs_inbox:
            base.subscribe_ml.delay(context, otrs_inbox, maillist, sub_type.INBOX)
            base.unsubscribe_ml.delay(context, otrs_mail, maillist, sub_type.INBOX)
        else:
            # если у такой рассылки не было otrs-подписчика, то подписываем otrs_email из ручки
            base.subscribe_ml.delay(context, otrs_email, maillist, sub_type.INBOX)
    else:
        # если такой рассылки еще не существует (maillist is None),
        # то создаем и подписываем otrs-email
        destinations = [otrs_email, ]
        base.create_alias_and_subscribe_users.delay(context, email, destinations, [])


def handling_service_separator(email):
    separator_email = 'separator@separator.yandex-team.ru'

    def _has_separator_subscriber(maillist):
        if maillist is None:
            return False
        elif maillist.emailsubscriber_set.filter(email=separator_email):
            return True
        return False

    maillist = get_maillist_by_email(email)
    context = {
        'initiator': 'robot_separator_emails',
        'comment': 'Change or create redirect: %s -> %s' % (email, separator_email),
    }
    exist_destination = _has_separator_subscriber(maillist)
    if maillist and exist_destination:
        return
    elif maillist:
        base.subscribe_ml.delay(context, separator_email, maillist, sub_type.INBOX)
    else:
        # если такой рассылки еще не существует (maillist is None),
        # то создаем и подписываем separator-email
        destinations = [separator_email, ]
        base.create_alias_and_subscribe_users.delay(context, email, destinations, [])

