# -*- coding: utf-8 -*-

from django.core.management.base import BaseCommand
from optparse import make_option
from django.conf import settings
from mlcore.aliases.utils import filter_crm_users
from mlcore.aliases.handling import handling_service_crm
from mlcore.aliases.utils import load_xml, ResponseException
from mlcore.aliases.models import Alias
from mlcore.aliases.handling import handling_service_otrs, get_maillist_by_email
from mlcore.subscribe import subscription_type as sub_type
from mlcore.tasks import base

import logging
logger = logging.getLogger(__name__)


# Подробнее, как работает CRM: https://wiki.yandex-team.ru/crm/tcrm/КакработаетпочтавCRM
def crm_users(url):
    tcrm_email = 'tcrm@yandex-team.ru'
    for email in filter_crm_users(url, tcrm_email):
        context = {
            'initiator': 'robot_crm_users',
            'comment': 'Create or(and) subscribe %s to tcrm@yandex-team.ru' % email,
        }
        handling_service_crm(context, email, tcrm_email)


def otrs(url, token):
    def _parse_xml(xml):
        for el in xml.iterfind('emails/email'):
            yield el.text, el.get('forward')

    try:
        xml = load_xml(url, token)
    except ResponseException as exc:
        logging.error(exc.message)
        return

    if not xml:
        logging.warning("Empty response from otrs-api")
        return

    existing_service_aliases = {a.virtual_email: a for a in Alias.objects.all() if a.destination.startswith('otrs@')}
    modified_maillists = set()

    for email, otrs_email in _parse_xml(xml):
        otrs_email = otrs_email.strip()
        email = email.strip()
        redirect = existing_service_aliases.pop(email, None)
        if redirect is not None:
            is_new = False
        else:
            redirect = Alias(virtual_email=email, destination=otrs_email)
            is_new = True

        if is_new or redirect.destination != otrs_email:
            redirect.destination = otrs_email
            redirect.save()
            handling_service_otrs(email, otrs_email)
            modified_maillists.add(email)

    # Отписываем от рассылки otrs-email-ы, если такие рассылки не вернули по OTRS_URL
    for email in existing_service_aliases:
        a = existing_service_aliases[email]
        modified_maillists.add(a.virtual_email)
        a.delete()
        context = {
            'initiator': 'otrs_robot_email',
            'message': 'unsubscribe %s from %s' % (a.destination, a.virtual_email)
        }
        maillist = get_maillist_by_email(a.virtual_email)
        if maillist:
            base.unsubscribe_ml.delay(context, a.destination, maillist, sub_type.INBOX)

    # MAILDLV-1493 - надо обновлять кэш для nwsmtp
    for email in modified_maillists:
        maillist = get_maillist_by_email(email)
        if maillist:
            maillist.set_as_modified()


class Command(BaseCommand):
    help = 'Save CRM as aliases'

    option_list = BaseCommand.option_list + (
        make_option(
            '-c',
            '--crm',
            action='store_true',
            default=False,
            help='Force update crm'
        ),
        make_option(
            '-m',
            '--managers',
            action='store_true',
            default=False,
            help='Force update crm-managers'
        ),
       make_option(
            '-o',
            '--otrs',
            action='store_true',
            default=False,
            help='Force update otrs'
        ),
    )

    def handle(self, *args, **options):
        crm_users(settings.TCRM_URL_API + 'api/ml/crmaliases')
        otrs(settings.SAMSARA_OTRS_API, settings.SAMSARA_TOKEN)

