# coding: utf-8

import json
from django_api_auth.decorators import passport_or_token_auth
from django.http import HttpResponse, HttpResponseBadRequest
from django.views.decorators.http import require_POST
from mlcore.aliases.forms import CreateMaillistAlias
from mlcore.tasks.base import create_alias_and_subscribe_users
from mlcore.utils.getters import get_user
from django.core.exceptions import ObjectDoesNotExist


def _get_owners(destinations):
    owners = []
    for email in destinations:
        try:
            owners.append(get_user(email))
        except (TypeError, ObjectDoesNotExist) as exc:
            continue
    return owners


@require_POST
@passport_or_token_auth
def add_alias(request):
    """
    Добавляем рассылки и виртуальные адреса (алиасы) с точками в ML.
    Где заводим?
    1. cmail выключен
    2. Обзательно заводим в паспорте.
    3. Создаем в интерфейсе ML-я с точками.
    4. Не создаем в Ymail, не разрешаем подписываться в общую папку.
    В свойствах рассылки задаем:
        - is_sub = False - т.е. не разрешаем подписаться в общую папку
        - внешняя (писать в рассылку могут все)
        - приватная (подписываться не всем разрешено)
        - email-подписчик - passport_name@mail.yandex-team.ru
        - запрашиваемые email-подписчики из параметра destinations
    """
    form = CreateMaillistAlias(request.POST)
    message = ''
    if form.is_valid():
        try:
            email = form.cleaned_data['email']
            destinations = form.cleaned_data['destinations']
            owners = _get_owners(destinations)
            context = {
                'initiator': owners[0].email if owners else 'robot',
                'comment': 'Create %s and subscribe users' % email,
            }
            create_alias_and_subscribe_users.delay(context, email, destinations, owners)
            return HttpResponse(json.dumps({
                'status': 'ok',
            }), content_type='application/json')
        except Exception as exc:
            message = "%s" % exc

    if form.errors:
        message = form.errors.as_text()

    errors = {'status': 'error', 'message': message}
    return HttpResponseBadRequest(json.dumps(errors),
                                  content_type='application/json')