# coding: utf-8
import json
from django import forms
from django.http import HttpResponse, HttpResponseBadRequest
from django.views.decorators.http import require_GET, require_POST
from django_api_auth.decorators import passport_or_token_auth

from mlcore.ml.models import MailList, SuidLookup, User, Owner
from mlcore.permissions.utils import (get_permitted_folders, can_read, can_write)
from mlcore.permissions.models import Type, ListPermission
from .base import MailListFormMixin, UIDFormMixin

def check_user_permission(username, maillist):
    return {'read': can_read(username, maillist),
            'write': can_write(username, maillist)}


class CheckRightsForm(forms.Form):
    user_suid = forms.IntegerField()
    folder_suid = forms.IntegerField()


class ChangePermissionForm(MailListFormMixin, UIDFormMixin, forms.Form):
    p_type = forms.CharField()

    def clean_p_type(self):
        type_name = self.cleaned_data['p_type']
        try:
            p_type = Type.objects.get(name=type_name)
        except Type.DoesNotExist:
            raise forms.ValidationError('Invalid type')

        return p_type


class ChangeAdminsForm(MailListFormMixin, UIDFormMixin, forms.Form):
    pass


@require_GET
def check_rights(request):
    f = CheckRightsForm(request.GET)

    if f.is_valid():
        try:
            maillist = MailList.objects.get(fsuid=f.cleaned_data['folder_suid'])
            login = SuidLookup.objects.get(suid=f.cleaned_data['user_suid']).login
        except (MailList.DoesNotExist, SuidLookup.DoesNotExist):
            return HttpResponseBadRequest("Cant't find user or list",
                                          content_type='text/plain; charset=UTF-8')
        data = check_user_permission(login, maillist)
        return HttpResponse(json.dumps(data),
                            content_type='application/json')
    else:
        return HttpResponseBadRequest(f.errors.as_text(),
                                      content_type='text/plain; charset=UTF-8')


class UserRightsForm(forms.Form):
    user_suid = forms.IntegerField()


@require_GET
def user_rights(request):
    f = UserRightsForm(request.GET)
    if f.is_valid():
        try:
            login = SuidLookup.objects.get(suid=f.cleaned_data['user_suid']).login
        except SuidLookup.DoesNotExist:
            return HttpResponseBadRequest("Cant't find user")

        subscribed = set(MailList.objects.filter(
            subscribers__user__username=login).values_list(
                'fsuid', 'name', 'subscribers__is_imap'))
        allow_lists = set(get_permitted_folders(login).values_list(
            'fsuid', 'name', 'subscribers__is_imap'))

        def get_data(item, subscribed):
            data = {'name': item[1], 'suid': item[0],
                    'permissions': {'read': True, 'write': True},
                    'subscribed': subscribed}
            if subscribed:
                if item[2]:
                    data['type'] = 'imap'
                else:
                    data['type'] = 'inbox'
            else:
                data['type'] = 'none'
            return data

        res = []
        for item in subscribed:
            res.append(get_data(item, True))

        for item in allow_lists - subscribed:
            res.append(get_data(item, False))

        return HttpResponse(json.dumps(res),
                            content_type='application/json')
    else:
        return HttpResponseBadRequest(f.errors.as_text(),
                                      content_type='text/plain; charset=UTF-8')


@require_POST
@passport_or_token_auth
def change_permission(request):
    form = ChangePermissionForm(request.POST)

    if form.is_valid():
        maillist = form.cleaned_data['maillist']
        permission_type = form.cleaned_data['p_type']
        uids = form.cleaned_data['uids']

        users = User.objects.filter(staff__uid__in=uids)
        for user in users:
            ListPermission.objects.grant_permission(maillist, user, permission_type)

        return HttpResponse(json.dumps({'status': 'ok'}),
                            content_type='application/json')

    errors = {'status': 'error', 'message': form.errors.as_text()}
    return HttpResponseBadRequest(json.dumps(errors), content_type='application/json')


@require_POST
@passport_or_token_auth
def change_administrators(request, action):
    form = ChangeAdminsForm(request.POST)

    if form.is_valid():
        maillist = form.cleaned_data['maillist']
        uids = form.cleaned_data['uids']

        users = User.objects.filter(staff__uid__in=uids)
        for user in users:
            if action == 'add':
                Owner.objects.get_or_create(list=maillist, user=user)
            elif action == 'delete':
                Owner.objects.filter(list=maillist, user=user).delete()

        return HttpResponse(json.dumps({'status': 'ok'}), content_type='application/json')

    errors = {'status': 'error', 'message': form.errors.as_text()}
    return HttpResponseBadRequest(json.dumps(errors), content_type='application/json')