# encoding: utf-8
import json
import logging

from django import forms
from django.http import HttpResponse
from django.utils.decorators import method_decorator
from django.views.generic import View

from mlcore.apiv2.views.helpers import maillist_by_email
from mlcore.apiv3.utils import tvm_required


class MailListFormMixin(forms.Form):
     maillist = forms.CharField()

     def clean_maillist(self):
        email = self.cleaned_data['maillist']

        maillist = maillist_by_email(email)
        if not maillist:
            raise forms.ValidationError('Mail list not found')

        return maillist


class JSONField(forms.CharField):
    def to_python(self, value):
        try:
            return json.loads(value)
        except (ValueError, TypeError):
            raise forms.ValidationError('Wrong json format')


class APIException(Exception):
    """APIView raises exception to signal premature end of request handling"""
    def __init__(self, http_status_code, message='error', details=None):
        self.http_status_code = http_status_code
        self.message = message
        self.details = details

    def as_http_response(self):
        """Turn exception into HTTP response object"""
        text = json.dumps({
            'status': 'error',
            'message': self.message,
            'details': self.details,
        })
        return HttpResponse(text, status=self.http_status_code, content_type='application/json')

    @classmethod
    def from_form_errors(cls, form, http_status_code=400):
        errors = form.errors
        common_errors = errors.pop('__all__', [])
        return cls(
            http_status_code=http_status_code,
            message='Validation error',
            details={'parameter_errors': errors, 'errors': common_errors}
        )

    def __str__(self):
        return "APIException<{}: {}>".format(self.http_status_code, self.message)


class APIView(View):
    """Base class for APIView - checks TVM, turns APIExceptions to uniformly formatted http responses"""

    def dispatch(self, request, *args, **kwargs):
        try:
            return super(APIView, self).dispatch(request, *args, **kwargs)
        except APIException as exc:
            return exc.as_http_response()

    @staticmethod
    def get_object_or_404(queryset, *args, **kwargs):
        try:
            return queryset.get(*args, **kwargs)
        except queryset.model.DoesNotExist:
            raise APIException(
                http_status_code=404,
                message='{name} not found'.format(name=queryset.model._meta.object_name)
            )

    @staticmethod
    def json_response(data, status=200):
        return HttpResponse(json.dumps(data), status=status, content_type='application/json')
