# encoding: utf-8

import json
import logging

from django.http import Http404
from django import forms
from django.db.models import Q
from django.http import HttpResponse, HttpResponseBadRequest
from django.utils.decorators import method_decorator
from django.views.decorators.http import require_GET, require_POST
from django_intranet_stuff.models import Staff

from mlcore.apiv3.views.base import APIView, APIException
from mlcore.ml.class_views.create_request import get_domain
from mlcore.ml.models import MailList, DEFAULT_DOMAIN, User, EmailSubscriber
from mlcore.ml.forms import CreateRequestForm
from mlcore.ml.queries import get_permitted_maillists
from mlcore.subscribe import subscription_type
from mlcore.ml.views import _has_circle
from mlcore.tasks import operations, base
from mlcore.permissions.utils import get_permitted_staff
from mlcore.nwsmtp_connect.utils import get_passport_aliases, get_maillist_by_email
from mlcore.subscribe.backends.yandex_team.models import YandexTeamBackendContext

from .base import MailListFormMixin, JSONField
from ..utils import tvm_required


LIST_INFO_VALID_FIELDS = {'name', 'is_open', 'email', 'info', 'exchange_name', 'id'}
LIST_INFO_DEFAULT_FIELDS = ('name', 'is_open', 'email', 'info')


def maillist_by_email(email):
    l = MailList.objects.filter(email=email).filter(is_deleted=False)
    if l:
        return l[0]


def _list_subscribers(maillist, cache=None, via=None, expand=True):

    if cache is None:
        cache = set()

    if maillist.id in cache:
        raise StopIteration

    cache.add(maillist.id)

    for username, email, is_imap, is_sub in maillist.subscribers_set\
            .filter(user__is_active=True)\
            .values_list('user__username', 'user__email', 'is_imap', 'is_sub'):
        data = {'login': username, 'inbox': is_sub, 'imap': is_imap, 'email': email}
        if via:
            data['via'] = via[:]
        yield data

    for email in maillist.emailsubscriber_set.values_list('email', flat=True):
        ml = maillist_by_email(email)
        if ml and expand:
            if via is None:
                via = []
            # Случай "рассылка-подписана-на-рассылку"
            via.append(email)
            for _ in _list_subscribers(ml, cache=cache, via=via, expand=expand):
                yield _
        else:
            data = {'email': email, 'login': None, 'inbox': True, 'imap': False}
            if via:
                data['via'] = via[:]
            yield data


class SubscribeForm(MailListFormMixin, forms.Form):
    subscribers = JSONField()

    def clean_subscribers(self):
        recipients = self.cleaned_data['subscribers']
        types = (subscription_type.IMAP, subscription_type.INBOX, subscription_type.BOTH)

        cleaned_subscribers = []
        for type_ in types:
            uids = recipients.get(type_)
            if uids:
                cleaned_subscribers.append((type_, uids))

        return cleaned_subscribers



@tvm_required
def lists_info(request):
    """
    Cписок всех рассылок с описанием.
    include_unsub – параметр который позволяет запросить все рассылки
        в т.ч. на которые нельзя подписаться
    fields - параметр позволяет выбрать определённые поля, которые будут в выдаче
    """

    include_unsub = request.REQUEST.get('include_unsub', False)
    data = MailList.objects.exclude_deleted()

    # Удалим из выдачи скрытые рассылки ML-1303
    data = data.filter(is_hidden=False)

    fields = request.REQUEST.get('fields', None)
    if fields:
        fields = list(LIST_INFO_VALID_FIELDS.intersection(fields.split(',')))

    if not fields:
        fields = LIST_INFO_DEFAULT_FIELDS

    if not include_unsub:
        data = data.exclude(is_sub=False, is_imap=True)

    # was: maillists = list(data.values( *fields ))
    # new version is slower, but can handle 'exchange_name' property

    maillists = []
    for d in data:
        row = {}
        for k in fields:
            row[k] = getattr(d, k, None)
        maillists.append(row)

    string = json.dumps({
        'maillists': maillists
    })
    return HttpResponse(string, mimetype="text/plain")


@require_GET
@tvm_required
def get_default_lists(request):
    data = {
        'maillists': list(MailList.objects.filter(email__endswith=DEFAULT_DOMAIN).values_list('email', flat=True)),
    }
    return HttpResponse(json.dumps(data),
                        content_type='application/json')


class CreateRequestApiForm(CreateRequestForm, forms.Form):
    def clean_responsible(self):
        data = self.cleaned_data['responsible']

        return data


@require_POST
@tvm_required
def create(request):
    form = CreateRequestApiForm(request.POST)

    if form.is_valid():
        # проверяем домен у себя в ML
        domain, error = get_domain(form.cleaned_data['name'])

        if not domain:
            errors = {'status': 'error', 'code': 'bad_domain', 'message': 'error'}
            return HttpResponseBadRequest(json.dumps(errors), content_type='application/json')

        data = form.cleaned_data.copy()

        operations.create_maillist.delay(
            {
                'initiator': 'api_subscription',
                'comment': data['name']
            },
            data=data,
            use_cmail=False
        )

        return HttpResponse(json.dumps({'status': 'ok'}), content_type='application/json')

    errors = {'status': 'error', 'message': form.errors.as_text()}
    return HttpResponseBadRequest(json.dumps(errors), content_type='application/json')


@require_POST
@tvm_required
def subscribe(request):
    form = SubscribeForm(request.POST)

    if form.is_valid():
        maillist = form.cleaned_data['maillist']
        for stype, uids in form.cleaned_data['subscribers']:
            user_list = User.objects.filter(staff__uid__in=uids)

            for user in user_list:
                query_kwargs = dict(
                    list=maillist
                )
                if stype == subscription_type.IMAP:
                    query_kwargs.update(is_imap=True)
                elif stype == subscription_type.INBOX:
                    query_kwargs.update(is_sub=True)
                elif stype == subscription_type.BOTH:
                    query_kwargs.update(is_sub=True, is_imap=True)
                if user.subscriptions.filter(**query_kwargs).exists():
                    continue

                operations.subscribe_user.delay({
                        'initiator': 'api_subscription',
                        'comment': u'Подписка через API #ML-1564'
                    },
                    user, maillist, stype, check_rights=False
                )

            # Подписываем рассылки
            if stype == subscription_type.INBOX:
                maillists = MailList.objects.filter(backends__yandexteambackendcontext__uid__in=uids)
                for s_maillist in maillists:
                    if (
                            s_maillist.name == maillist.name or
                            _has_circle(maillist, s_maillist) or
                            _has_circle(s_maillist, maillist)
                    ):
                        continue

                    es, created = EmailSubscriber.objects.get_or_create(email=s_maillist.email, list=maillist)
                    if created:
                        context = {
                            'initiator': 'api_subscription',
                            'comment': u'Подписка рассылки %s на рассылку %s API #ML-1564' %
                                       (s_maillist.email, maillist.name)
                        }
                        base.subscribe_inbox(context, user=s_maillist.email, maillist=maillist.name).delay()

        return HttpResponse(json.dumps({'status': 'ok'}), content_type='application/json')

    errors = {'status': 'error', 'message': form.errors.as_text()}
    return HttpResponseBadRequest(json.dumps(errors), content_type='application/json')


@require_POST
@tvm_required
def unsubscribe(request):
    form = SubscribeForm(request.POST)

    if form.is_valid():
        maillist = form.cleaned_data['maillist']
        for stype, uids in form.cleaned_data['subscribers']:
            user_list = User.objects.filter(staff__uid__in=uids)

            for user in user_list:
                query_set = user.subscriptions.filter(list=maillist)
                unsubscribe_task = base.unsubscribe_both

                if stype == subscription_type.IMAP:
                    query_set = query_set.filter(is_imap=True)
                    unsubscribe_task = base.unsubscribe_imap
                elif stype == subscription_type.INBOX:
                    query_set = query_set.filter(is_sub=True)
                    unsubscribe_task = base.unsubscribe_inbox
                elif stype == subscription_type.BOTH:
                    query_set = query_set.filter(Q(is_sub=True) | Q(is_imap=True))

                if query_set.exists():
                    unsubscribe_task({
                        'initiator': 'api_subscription',
                        'comment': u'Отписка через API #ML-1564'
                    }, user, maillist).delay()

            # Отписываем рассылки
            if stype == subscription_type.INBOX:
                maillists = MailList.objects.filter(backends__yandexteambackendcontext__uid__in=uids)
                for s_maillist in maillists:
                    try:
                        es = EmailSubscriber.objects.get(email=s_maillist.email, list=maillist)
                        es.delete()

                        context = {
                            'initiator': 'api_subscription',
                            'comment': u'Отписка рассылки %s от рассылки %s #ML-1564' %
                                       (s_maillist.email, maillist.name)
                        }
                        base.unsubscribe_inbox(context, user=s_maillist.email, maillist=maillist.name).delay()
                    except EmailSubscriber.DoesNotExist:
                        continue

        return HttpResponse(json.dumps({'status': 'ok'}), content_type='application/json')

    errors = {'status': 'error', 'message': form.errors.as_text()}
    return HttpResponseBadRequest(json.dumps(errors), content_type='application/json')


@require_GET
@tvm_required
def lists_get(request):
    """
    Получить информацию о рассылке по email.
    email – email запрашиваемой рассылки
    """
    form = MailListFormMixin(request.GET)
    if form.is_valid():
        maillist = form.cleaned_data['maillist']
        responsible = [owner.user.email for owner in maillist.owner_set.all()]

        string = json.dumps({
            'name': maillist.name,
            'info': maillist.info,
            'info_en': maillist.info_en,
            'responsible': responsible,
            'is_internal': maillist.is_internal,
        })
        return HttpResponse(string, content_type='application/json')
    else:
        raise Http404("Maillist not found")


@tvm_required
@require_GET
def lists_subscribers(request):
    """
    ML-1396: Ручка экспорта подписчиков рассылки для Календаря
    """
    emails = request.GET.get('emails')
    expand = request.GET.get('expand', 'no') == 'yes'
    if emails:
        r = {}
        emails = emails.split(',')
        for email in emails:
            maillist = maillist_by_email(email)
            if maillist:
                data = {'subscribers': list(_list_subscribers(maillist, expand=expand)),
                        'is_internal': maillist.is_internal,
                        'is_open': maillist.is_open,
                        }
                if maillist.readonly:
                    data.update({'readonly': True,
                                 'who_can_write': get_passport_aliases(
                                     set(get_permitted_staff(maillist, 'write')
                                        .filter(is_dismissed=False).values_list('work_email', flat=True))),})

                r[email] = data
            else:
                r[email] = None
        return HttpResponse(json.dumps({'result': r}), content_type='application/json')
    else:
        return HttpResponse("emails parameter required", status=400)


@tvm_required
@require_GET
def get_corp_user(request):
    # ML-1776 Возможность резолвить внешнюю рассылку которая замаплена на внутреннюю
    email = request.GET.get('email', '')
    maillist = get_maillist_by_email(email)
    if not maillist:
        raise Http404

    try:
        backend = YandexTeamBackendContext.objects.get(backend_type='yandex_team', maillist=maillist)
    except YandexTeamBackendContext.DoesNotExist:
        raise Http404

    return HttpResponse(json.dumps({'result': {'email': '{}@yandex-team.ru'.format(backend.passport_name),
                                    'uid': backend.uid}}),
                        content_type='application/json')


class GetStaffParamsForm(forms.Form):
    """Validates parameters which can be used to identify staff member"""
    uid = forms.IntegerField(required=False, min_value=0)
    login = forms.CharField(required=False)
    work_email = forms.EmailField(required=False)

    def clean(self):
        """Get only non-empty validated parameters (to get staff entity)"""
        data = super(GetStaffParamsForm, self).clean()
        one_of_fields_is_required = ['uid', 'work_email', 'login']
        if all(field not in self.data for field in one_of_fields_is_required):
            raise forms.ValidationError(
                'At least one of {fields} parameters is required'.format(fields=one_of_fields_is_required))
        return {key: value for key, value in data.items() if value}


class ListReadableMailListsForStaffView(APIView):
    """Get UIDs of maillists which given staff can read"""

    @method_decorator(tvm_required)
    def get(self, request):
        user = self._get_user(request)
        response_data = get_permitted_maillists(user)
        return self.json_response(response_data)

    def _get_user(self, request):
        form = GetStaffParamsForm(request.GET)
        if not form.is_valid():
            raise APIException.from_form_errors(form)
        staff = self.get_object_or_404(Staff.objects, **form.cleaned_data)

        if not staff.user:
            error_msg = 'There is no User for Staff {staff_id}'.format(staff_id=staff.id)
            logging.error(error_msg)
            raise APIException(http_status_code=500, message=error_msg)

        return staff.user
