# encoding: utf-8
from django.conf import settings
from django.forms import forms, fields
from django.utils.decorators import method_decorator

from mlcore.apiv3.utils import tvm_required
from mlcore.apiv3.views.base import APIView, APIException
from mlcore.utils.domain_name import Domain, DomainError
from ya_directory import api as directory_api, DirectoryAPIError, DirectoryAlreadyExistsError, DirectoryNotFound


class CreateUserInOrganizationForm(forms.Form):
    domain = fields.CharField(required=True)
    nickname = fields.CharField(required=True)
    first_name_ru = fields.CharField(required=False)
    first_name_en = fields.CharField(required=False)
    password = fields.CharField(required=True)

    def clean_domain(self):
        try:
            domain = Domain(self.cleaned_data['domain'])
            domain.validate()
            return domain.name
        except DomainError as exc:
            raise forms.ValidationError(exc.description)


class GetUserInOrganizationForm(forms.Form):
    email = fields.EmailField(required=True)


class CreateUserInOrganizationView(APIView):
    """Внутрення ручка для Рассылятора для создания пользователей в желаемом домене для сбора баунсовых писем.
    По домену создаем или находим организацию и в ней создаем пользователей с желаемым алиасом и паролем.

    Рассылятору нуже обычный юзер, в которого можно залогиниться через imap.yandex.ru
    для сбора баунсовых имейлов аккаунта. Ящик этого юзера будет указываться в SMTP команде MAIL FROM,
    позже по IMAP будут собираться отлупы почтовых систем, возвращающих сообщения о недоставках.

    Так же создаем ящики для FBL отчетов.
    """

    def _raise_from_directory_error(self, exc):
        raise APIException(
            http_status_code=500,
            message=exc.message or 'Error while operating on Directory API',
            details={
                'directory_response': (
                    {
                        'status_code': exc.response_status_code,
                        'content': exc.response_content,
                        'data': exc.response_data,
                    } if exc.response else None
                ),
                'error': exc.message,
                'error_code': exc.code,
            },
        )

    @method_decorator(tvm_required(allowed_clients=(settings.SENDER_TVM2_ID,)))
    def post(self, request):
        """
        Создаем юзера вместе с организацией в желаемом домене, а если юзер существует возвращаем 409 с данными юзера
        """
        form = CreateUserInOrganizationForm(request.POST)
        if not form.is_valid():
            raise APIException.from_form_errors(form)
        data = form.cleaned_data

        directory_operations = directory_api.DirectoryMaillistsOperations()
        try:
            try:
                user_data = directory_operations.create_user_for_domain(
                    domain_name=data['domain'],
                    nickname=data['nickname'],
                    password=data['password'],
                    first_name_en=data['first_name_en'],
                    first_name_ru=data['first_name_ru'],
                )
                return self.json_response({'status': 'OK', 'result': {'user': user_data}}, status=201)
            except DirectoryAlreadyExistsError:
                # ожидается, что DirectoryAlreadyExistsError будет если существует именно юзер
                user_data = directory_operations.get_user_in_domain(data['domain'], data['nickname'])
                raise APIException(
                    http_status_code=409,
                    message='User already exists',
                    details={
                        'user': user_data,
                    }
                )
        except DirectoryAPIError as exc:
            self._raise_from_directory_error(exc)

    @method_decorator(tvm_required(allowed_clients=(settings.SENDER_TVM2_ID,)))
    def get(self, request):
        """Чтобы Рассылятор мог проверить наличие пользователя"""
        form = GetUserInOrganizationForm(request.GET)
        if not form.is_valid():
            raise APIException.from_form_errors(form)

        email = form.cleaned_data['email']
        nickname, domain = email.split('@', 1)

        directory_operations = directory_api.DirectoryMaillistsOperations()
        try:
            try:
                user_data = directory_operations.get_user_in_domain(domain_name=domain, nickname=nickname)
                return self.json_response({'status': 'OK', 'result': {'user': user_data}}, status=200)
            except DirectoryNotFound:
                raise APIException(
                    http_status_code=404,
                    message='User not found',
                    details='No such user {email}'.format(email=email),
                )
        except DirectoryAPIError as exc:
            self._raise_from_directory_error(exc)
