# -*- coding: utf-8 -*-
from email.Header import make_header

from django.template import loader, Context
from django.utils import translation
from django.conf import settings
from django.core.mail import EmailMessage

from django_intranet_stuff.models import Staff


def get_groupped_by_lang_ui_recipients(recipients):

    def short_language(language):
        if '-' in language:
            language = language.split('-')[0]
        return language

    groupped_by_lang_ui_recipients = {}
    for recipient in recipients:
        try:
            staff = Staff.objects.get(work_email=recipient)
        except Staff.DoesNotExist:
            lang = settings.LANGUAGE_CODE
        else:
            lang = staff.lang_ui
        finally:
            group = groupped_by_lang_ui_recipients.setdefault(short_language(lang), [])
            group.append(recipient)

    return groupped_by_lang_ui_recipients


class Notification(object):
    template_type = None
    in_reply = False
    content = 'html'

    def __init__(self, context=None):
        self.context = context if context is not None else {}

    def send(self, recepients, sender=None, cc=None, reply_to=None):
        recipients = list(recepients)
        recipients.append(settings.EMAIL_COPY)
        groupped_by_lang_ui_recipients = get_groupped_by_lang_ui_recipients(recipients)
        for lang in groupped_by_lang_ui_recipients.keys():
            group = groupped_by_lang_ui_recipients[lang]

            cur_language = translation.get_language()
            try:
                translation.activate(lang)
                msg = self.get_message(to=group, sender=sender, cc=cc,
                                       reply_to=reply_to)
                msg.send()
            finally:
                translation.activate(cur_language)

    def get_message(self, to=None, sender=None, cc=None, reply_to=None):
        if to is None:
            to = []

        self.context['in_reply'] = self.in_reply
        self.context['base_url'] = settings.EMAIL_BASE_URL.rstrip('/')

        if not sender:
            sender = settings.SERVICE_EMAIL
            self.context['do_not_reply'] = True
        else:
            self.context['do_not_reply'] = sender == settings.SERVICE_EMAIL

        if settings.DEBUG and (not settings.DEBUG_EMAIL in to or len(to) > 1):
            self.context['original_emails'] = to
            to = [settings.DEBUG_EMAIL]

        self.context['LANGUAGE_CODE'] = translation.get_language()
        template = loader.get_template('notifications/%s' % self.template_type)
        content = template.render(Context(self.context))

        subject, content_body = content.split('\n\n', 1)

        msg = EmailMessage(subject, content_body, from_email=sender, to=to, cc=cc)

        if reply_to:
            msg.extra_headers['Reply-To'] = self.get_mail_header(reply_to)

        msg.extra_headers['From'] = self.get_mail_header(sender)

        msg.encoding = "UTF-8"
        msg.content_subtype = self.content
        return msg

    def get_mail_header(self, data):
        if isinstance(data, (list, tuple)) and len(data) == 2:
            hdr = make_header([(data[0].encode('utf-8'), "utf-8"),
                               ("<" + data[1] + ">", "us-ascii")])
        elif isinstance(data, basestring):
            hdr = data
        else:
            raise ValueError("Email may be only string or tuple, not %s" % type(data))
        return hdr
