# -*- coding:utf-8 -*-
"""
Про текущий способ взаимодействия с базой в ML.

В проекте используется ``django_replicated.middleware.ReplicationMiddleware``,
который явно устанавливает на роутере начальное состояние:
>>> database_router.init(state)

Для входящих HTTP запросов с методами GET и HEAD ReplicationMiddleware установит роутере в состояние 'slave',
а для остальных запросов - в состояние 'master'.

Другими словами, подразумевается распространенное соглашение,
что обработка запросов с методами HEAD и GET не предполагает изменения состояние базы,
а обработка запросов с методами POST, PUT, PATCH допускает запись в базу.

Будучи в состоянии 'master', роутер консистентно определяет дефолтную базу для чтения и записи,
в том смысле, что чтение и запись будут происходить на соединении django.db.connections['default'],
в частности корректно будут обрабатываться контекст transaction.atomic.

Предполагается, что база settings.DATABASES['default'] - это ведущий сервер БД.

В состоянии 'slave' для "читающих" запросов роутер попытается сначала выбрать реплику, и только в случае
невозможности такого выбора определит дефолтную базу.

Теоретически, в состоянии 'slave' роутер не позволяет консистентно использовать Django Query API,
так как пишущие операции будут работать с ведущим сервером (например, Model.object.create),
а чтение может происходить из реплики (например, Model.objects.filter).

Однако на практике такой проблемы нет, поскольку во время обработки запросов ``ReplicationMiddleware``
явно укажет состояние 'master' на роутере для запросов с методами POST, PUT, PATCH, при условии,
что мы придерживаемся соглашения о методах HTTP запросов, либо явно указываем состояние 'master',
если по какой-то причине вынуждены нарушить соглашение.

Аналогично нет проблемы с кодом, работающим в рамках асинхронных задач: при старте приложения
роутер не присваивает себе явно никакого состояния, а по-умолчанию роутер считает себя в состоянии 'master'.

@todo Разобраться с ATOMIC_REQUESTS

Настройка ATOMIC_REQUESTS и кастомный DatabaseRouter не очень дружат вместе, так как
если ATOMIC_REQUESTS=True, то попытка войти в транзакцию и установить соединение с базой происходит внутри
Django в базовом обработчике запросов, но ТОЛЬКО database_router знает, на какой базе в settings.DATABASES
нужно начать транзакцию. Например, для обработки GET запросов нужно сделать что-то похожее на:
>>> alias = database_router.db_for_read(...)
>>> with transaction.atomic(alias)
>>>     pass # some code

Базовая логика Django начинает транзакцию на всех базах, у которых указана настройка ATOMIC_REQUESTS.
Из-за этого, если мастер или реплика будут недоступны, ML не уйдет в read-only режим, как можно было бы ожидать,
а начнет сыпать 500-ми на обычные GET запросы, пытаясь начать транзакцию с недоступным хостом.
Очевидно, что если уходит один их хостов, то на нем не нужно пытаться начать транзакцию.
Точнее говоря, нет смысла вообще начинать транзакцию на нескольких хостах, так как роутер выберет ровно один.

Нельзя просто убрать настройку ATOMIC_REQUESTS (даже только у реплики),
ограничившись явным указанием transaction.atomic только в обработчиках мутирующих запросов.
Удаление ATOMIC_REQUESTS эквивалентно понижению уровня изоляции читающих транзакций
до уровня READ COMMITTED, поскольку в MySQL 5.7 (с InnoDB) по-умолчанию уровень изоляции транзакций REPEATABLE READ.
Чтобы сохранить текущее поведение АПИ нужно начинать транзакцию во время обработки GET запросов,
чтобы происходила работа со снепшотом состояния базы.

Вариант 1 - научить базовую логику обработки запросов консультироваться с database_router, чтобы
знать наверняка, на которой из баз в settings.DATABASES начинать транзакцию. Так удастся сохранить
уровень изоляции REPEATABLE READ для GET запросов и исключить проблемный хост.

Вариант 2. Если уровень изоляции READ COMMITED приемлем для обработки GET запросов, тогда можно убрать ATOMIC_REQUESTS
из настроек баз, избавившись от необходимости согласовывать работу базовых вью Джанги с database_router.
Но нужно будет явно обернуть в контекст transaction.atomic всю мутирующую логику в обработчиках.
С одной стороны, ReplicationMiddleware для мутирующих запросов проставит роутеру состояние master,
что эквивалентно работе с default базой.
С другой стороны, transaction.atomic начнет по-умолчанию транзакцию на default базе.
Таким образом дополнительных мер по согласованию логики роутера с транзакциями не потребуется.
"""

import logging
import random
import socket
import time
from datetime import datetime, timedelta
from functools import partial, wraps
from threading import Event, RLock, Thread

from django.conf import settings
from django.core.cache import DEFAULT_CACHE_ALIAS, get_cache
from django.db import connections
from django.db.utils import DatabaseError
from django_replicated.router import ReplicationRouter

logger = logging.getLogger('replicated.failover_router')

DATABASE_ERRORS = (DatabaseError,)

try:
    import MySQLdb as mysql

    DATABASE_ERRORS += (mysql.DatabaseError,)
except ImportError:
    pass

try:
    import psycopg2 as pg

    DATABASE_ERRORS += (pg.DatabaseError,)
except ImportError:
    _pg_database_errors = ()  # noqa


def _close_db_connection(connection):
    if connection:
        try:
            connection.close()
        except DATABASE_ERRORS as exc:
            # ProgrammingError: closing a closed connection
            if "closed" not in str(exc):
                raise


def _db_is_alive(db_name):
    """
    mysql бэкэнду после сбоя нужно помочь с закрытием соединения с базой.
    django-replicated этого не делает, поэтому мы делаем это сами.

    Воспроизвести проблему:
    1) закрыть мастер через iptables
    2) открыть IPython шелл, попробовать поработать с connections['default'], например:
        >>> MailList.objects.last()
    3) получить ошибку
        OperationalError: (2006, 'MySQL server has gone away')
    4) открыть мастер через iptables
    5) попробовать снова обратиться к connections['default']
    6) снова получим ошибку
        OperationalError: (2006, 'MySQL server has gone away')

    Соответственно нужно после сбоя:
    1) connections['default'].connection.close() => bool(connections['default'].open) будет False
    2) установить connections['default'].connection = None,
        чтобы при следующем обращении к базе создать новое соединение
    """

    db = connections[db_name]
    try:
        if db.connection is not None and not db.connection.open:
            # Сбрасываем атрибут соединения на DatabaseWrapper,
            # чтобы соединение было открыто заново.
            db.connection = None

        if db.connection is not None and hasattr(db.connection, 'ping'):
            logger.debug(u'Ping db %s.', db_name)
            db.connection.ping()
        else:
            logger.debug(u'Get cursor for db %s.', db_name)
            db.cursor()
        return True
    except Exception as exc:
        logger.exception(u'Error (%s) while verifying db %s. Connection is: %s', exc, db_name, db.connection)
        _close_db_connection(db.connection)  # Исправление
        db.connection = None
        return False


cache = get_cache(
    getattr(settings, 'REPLICATED_CACHE_BACKEND', DEFAULT_CACHE_ALIAS)
)
host_name = socket.gethostname()


def check_db(checker, db_name, cache_seconds=0, number_of_tries=1, force=False):
    assert number_of_tries >= 1, u'Number of tries must be >= 1.'

    cache_td = timedelta(seconds=cache_seconds)

    checker_name = checker.__name__
    cache_key = host_name + checker_name

    check_cache = cache.get(cache_key, {})

    death_time = check_cache.get(db_name)
    if death_time:
        if death_time + cache_td > datetime.now():
            logger.debug(
                u'Last check "%s" %s was less than %d ago, no check needed.',
                checker_name, db_name, cache_seconds
            )
            if not force:
                return False
            logger.debug(u'Force check "%s" %s.', db_name, checker_name)

        else:
            del check_cache[db_name]
            logger.debug(
                u'Last check "%s" %s was more than %d ago, checking again.',
                db_name, checker_name, cache_seconds
            )
    else:
        logger.debug(
            u'%s cache for "%s" is empty.',
            checker_name, db_name
        )

    is_alive = False
    for count in range(1, number_of_tries + 1):
        is_alive = checker(db_name)
        logger.debug(
            u'Trying to check "%s" %s: %d try.',
            db_name, checker_name, count
        )
        if is_alive:
            logger.debug(
                u'After %d tries "%s" %s = True',
                count, db_name, checker_name
            )
            break

    if not is_alive:
        msg = u'After %d tries "%s" %s = False.'
        logger.warning(msg, number_of_tries, db_name, checker_name)
        check_cache[db_name] = datetime.now()

    cache.set(cache_key, check_cache)

    return is_alive


db_is_alive = partial(check_db, _db_is_alive)


def use_state(state):
    """
    Создать декоратор, отмечающий приоритетную роль (master или slave), по которой попытаемся определить хост.
    Имеет смысл только для читающих запросов, так как кастомный роутер для пишущих запросов отдает базу default.

    Версия декоратора из django_replicated падает в тестинге, где не подключен ReplicatedRouter.
    Кроме того, она делает ненужную нам работу по проверке кук и настроек REPLICATED_VIEWS_OVERRIDES.
    Наша версия закостылена для избавления от этих недостатков.
    """

    def decorator(func):

        try:
            from django_replicated.utils import routers
        except AttributeError:
            return func

        @wraps(func)
        def wrapper(request, *args, **kwargs):
            routers.use_state(state)
            try:
                response = func(request, *args, **kwargs)
            finally:
                routers.revert()
            return response

        return wrapper

    return decorator


prefer_slave = use_state('slave')


class FailoverThread(Thread):
    """
    Для экземляра фейловер роутера периодически проверяем доступность реплик.
    Сообщаем роутеру о достуных и недоступных репликах, чтобы он мог выбирать базу корректно.
    """

    def __init__(self, router):
        super(FailoverThread, self).__init__()
        self.logger = logger.getChild(self.__class__.__name__)
        self.router = router
        self.check_interval = getattr(settings, 'DATABASE_ASYNC_CHECK_INTERVAL', 5)
        self._stop = Event()

    def join(self, timeout=None):
        self.logger.debug('Stop and join watcher thread...')
        self._stop.set()
        super(FailoverThread, self).join(timeout=timeout)

    def run(self):
        while not self._stop.isSet():
            self.check()
            time.sleep(self.check_interval)

    def check(self):
        self.check_slaves()

    def check_slaves(self):
        """
        Для реплик, которые мы считаем доступными, проверяем доступность,
        и если реплика стала недоступна - помечаем ее как недоступную и убираем из роутера.
        Для реплик, которые мы посчитали недоступными, проверяем доступность,
        и если реплика стала доступной - помечаем ее как доступную и добавляем ее в роутер.
        """
        for alias in self.router.SLAVES:
            self.logger.debug('Check database %(alias)s still alive', {'alias': alias})
            if not self.db_is_alive(alias):
                self.router.deactivate_slave(alias)

        for alias in self.router.deactivated_slaves:
            self.logger.debug('Check database %(alias)s alive again', {'alias': alias})
            if self.db_is_alive(alias):
                self.router.activate_slave(alias)

    def db_is_alive(self, alias, **kwargs):
        return db_is_alive(alias, **kwargs)


class FailoverReplicationRouter(ReplicationRouter):
    """
    Базовый ReplicationRouter по текущему состоянию (self.state()) подбирает базу для пишущих и читающих запросов.
    Состояние роутера 'master':
        пишущие и читающие запросы следует направлять в 'default' базу.
    Состояние роутера 'slave':
        пишущие запросы следует направлять в 'default' базу,
        а читающие направлять в произвольную реплику из списка settings.DATABASE_SLAVES,
        либо в 'default', если список реплик пуст или реплики недоступны.

    Предполагается, что 'default' база является ведущей.

    Для ML интерфейс роутера расширили, чтобы можно было попросить роутер убрать или добавить реплику
    в список доступных для чтения реплик. Вспомогательный поток периодически проверяет доступность реплик
    и сообщает роутеру о доступных и недостпных репликах.
    """

    checker_cls = FailoverThread

    def __init__(self, run_thread=None, checker_cls=None):
        super(FailoverReplicationRouter, self).__init__()
        self.deactivated_slaves = []
        self.rlock = RLock()
        self.thread = None
        self.PREFERRED_SLAVES = set(getattr(settings, 'DATABASE_PREFERRED_SLAVES', []))
        self.logger = logger.getChild(self.__class__.__name__)

        if run_thread is not None:
            _run = run_thread
        else:
            _run = getattr(settings, 'DATABASE_ASYNC_CHECK', False)

        if _run and self.SLAVES:
            self.start_thread(cls=checker_cls)

    def start_thread(self, cls):
        cls = cls or self.checker_cls
        self.thread = cls(router=self)
        self.thread.start()
        self.logger.info('Start watcher thread')

    def stop_thread(self):
        if self.thread:
            logger.info("Stop watcher thread")
            self.thread.join(timeout=self.thread.check_interval * 2)
            logger.info("Watcher thread stopped")

    def deactivate_slave(self, alias):
        with self.rlock:
            if alias not in self.deactivated_slaves:
                self.logger.info(
                    "Deactivation: add database %(alias)s to dead slaves group %(group)s",
                    dict(alias=alias, group=self.deactivated_slaves),
                )
                self.deactivated_slaves.append(alias)
            if alias in self.SLAVES:
                self.logger.info(
                    "Deactivation: remove database %(alias)s from alive slaves group %(group)s",
                    dict(alias=alias, group=self.SLAVES),
                )
                self.SLAVES.remove(alias)

    def activate_slave(self, alias):
        with self.rlock:
            if alias not in self.SLAVES:
                self.logger.info(
                    "Activation: add database %(alias)s to alive slaves group %(group)s",
                    dict(alias=alias, group=self.SLAVES),
                )
                self.SLAVES.append(alias)
            if alias in self.deactivated_slaves:
                self.logger.info(
                    "Activation: remove database %(alias)s from dead slaves group %(group)s",
                    dict(alias=alias, group=self.deactivated_slaves),
                )
                self.deactivated_slaves.remove(alias)

    def __del__(self):
        self.stop_thread()

    def _get_db_for_read(self, model, **hints):
        state = self.state()

        if state == 'master':
            return self.db_for_write(model, **hints)

        # если уже был выбран хост для состояния 'slave',
        # т. к. состояния только 'master' и 'slave'.
        if state in self.context.chosen:
            db_alias = self.context.chosen[state]

            if connections[db_alias].in_atomic_block:
                # если база для чтения уже определена c ней начата транзакция
                # следует консистентно отдавать туже самую базу, даже если она вдруг стала недоступной
                # предполагаем, что код начинает транзакцию на дефолтной базе, либо использует роутер
                # для определния базы
                return db_alias

            if db_alias in self.SLAVES:
                # если соединение вне транзакции
                return db_alias

        # если с базой не начата транзакция, то допускаем подмену другой базы для чтения
        if self.SLAVES:
            preferred_slaves = None
            if self.PREFERRED_SLAVES:
                preferred_slaves = list(self.PREFERRED_SLAVES.intersection(self.SLAVES))
            chosen = random.choice(preferred_slaves or self.SLAVES)
        else:
            chosen = self.DEFAULT_DB_ALIAS

        self.context.chosen[state] = chosen
        return chosen

    def db_for_read(self, model, **hints):
        alias = self._get_db_for_read(model, **hints)
        self.logger.debug(
            'Database for read [model=%(model)s, state=%(state)s]: %(alias)s',
            dict(model=model.__name__, state=self.state(), alias=alias)
        )
        return alias

    def db_for_write(self, model, **hints):
        alias = super(FailoverReplicationRouter, self).db_for_write(model, **hints)
        self.logger.debug(
            'Database for write [model=%(model)s, state=%(state)s]: %(alias)s',
            dict(model=model.__name__, state=self.state(), alias=alias)
        )
        return alias
