# coding: utf-8

import logging
import time
from logging import getLogger

from requests import Response

from mlcore.utils.tvm2 import get_tvm_2_header
from .compat import compat_requests_session
from django.conf import settings


def _add_tvm_headers(kwargs, default_destination_id=None):
    """Pop destination_id and update headers with TVM tickets if destination_id is present"""
    destination_id = kwargs.pop('destination_id', default_destination_id)
    if destination_id:
        headers = kwargs.get('headers', {})
        headers.update(get_tvm_2_header(destination_id))
        kwargs['headers'] = headers
    return kwargs


def wrap_session(session, dry_run=False, logger=None, destination_id=None):
    """ Функция оборачивает сессию, чтобы она запоминала время работы """
    if logger is None:
        logger = logging.getLogger(__name__)

    req = session.request

    def inner(method, url, *args, **kwargs):
        logger.info(u"Вызов апи [%s] %s args=%s, kwargs=%s",
                    method, url, args, kwargs)

        kwargs = _add_tvm_headers(kwargs, default_destination_id=destination_id)

        start = time.time()
        res = req(method, url, *args, **kwargs)
        stop = time.time()
        # res.timings = {'start': start, 'stop': stop, 'total': stop - start}
        logger.info(u"Апи вернуло: [%s %s] %s, через %s сек", method, res.status_code,
                    res.text, stop - start)
        return res

    def dry(method, url, *args, **kwargs):
        logger.info(u"dry_run Вызов апи [%s] %s args=%s, kwargs=%s",
                    method, url, args, kwargs)

        # до проверяемых урлов нет доступов до тестинга (и скорее всего не будет)
        if settings.PASSPORT_URL in url:
            res = req(method, 'http://ya.ru', *args, **kwargs)
            res.status_code = 200
            return res

        if settings.YWMI_API_URL in url:
            return {"folders": {1120000000027049}}

        if settings.YORK_API_URL in url:
            res = Response()
            res.status_code = 200
            res._content = b'{"ok": true}'
            return res

        kwargs = _add_tvm_headers(kwargs, default_destination_id=destination_id)

        res = req(method, url, *args, **kwargs)
        res.status_code = 200
        return res

    if dry_run:
        session.request = dry
    else:
        session.request = inner
    return session


class BaseMailInteractionBackend(object):
    backend_type = 'base'

    def __init__(self):
        super(BaseMailInteractionBackend, self).__init__()
        self.log = getLogger(self.backend_type + '_client')
        self.conlog = getLogger('backends_connections')
        session = compat_requests_session()
        self.http = wrap_session(session)

    def error_test(self, response):
        # msg = "%s: started at %s, total %f, url %s"
        msg = "%s: url %s"
        args = [self.backend_type,
                # datetime.fromtimestamp(response.timings['start']),
                # response.timings['total'],
                response.url]

        if response.request and response.request.method == 'POST':
            msg += ', data: %s'
            args.append(response.request)  # .data

        self.conlog.info(msg, *args)

        if not (200 <= response.status_code < 300):
            message = "%s [%s] on url %s, %s"
            args = (response.request.method, response.status_code,
                    response.url, response.content)
            self.log.error(message, *args)
            raise IOError("%s [%s] on url %s" % args[:-1])
