# coding: utf-8
from __future__ import unicode_literals
"""
ML-1056
https://wiki.yandex-team.ru/users/dskut/message-body
"""

from .compat import compat_requests_session, compat_requests_json
from django.conf import settings

from base import BaseMailInteractionBackend, wrap_session


def returns_str(func):
    def wrapper(*a, **kw):
        r = func(*a, **kw)
        if isinstance(r, unicode):
            return r.encode('utf-8')
        return r
    return wrapper


class MessageBodyError(Exception):
    def __init__(self, message, code=None):
        self.message = message
        self.code = code
        super(MessageBodyError, self).__init__(message, code)


class MessageBodyBackend(BaseMailInteractionBackend):

    backend_type = 'messagebody'

    def __init__(self, url=settings.MESSAGE_BODY_URL, timeout=20,
                 dry_run=False, logger=None, destination_id=settings.MESSAGE_BODY_TVM_ID, **kwargs):
        super(MessageBodyBackend, self).__init__()

        if logger:
            self.log = logger

        session = compat_requests_session(timeout=timeout, **kwargs)
        self.http = wrap_session(session, dry_run=dry_run, logger=logger, destination_id=destination_id)
        self.url = url
        self.dry_run = dry_run

    def check_and_parse(self, response):
        try:
            self.error_test(response)
        except IOError:
            if response.status_code == 400:
                raise MessageBodyError("Bad request. %s" % response.url)
            else:
                raise
        json = compat_requests_json(response)
        return json

    def message(self, **kw):
        """
        Параметры:
        stid - мулечный идентификатор письма, по нему достаем исходник письма
        mid - оракловый идентификатор письма. Нужен для логгирования видео-ссылок и похода в томиту
        msg_date - время получения письма (unixtime). Нужен для логгирования видео-ссылок и похода в томиту
        subj - тема письма. Нужен для логгирования видео-ссылок и похода в томиту
        to - получатель письма, для системных писем, у к-рых нет хедера To
        uid - uid отправителя. Нужен для логгирования видео-ссылок, похода в томиту и оборачивания ссылок в обычных (не-смс) письмах
        from - адрес отправителя. Нужен для логгирования pkpass-аттачей и похода в томиту
        charset - кодировка письма. Передается, если его передала верстка. Если параметр передан, то именно эта кодировка берется для раскодирования письма (а не та, что указана в хедере)
        types - "томитные" типы письма (отели, авиабилеты, события итд) через запятую /pochta/types
        flags - специальные флаги, которые передает верстка (через запятую):
        IgnoreHighlight - не отправлять письмо в хайлайтер, преобразовать тело к html
        OutputAsCDATA - не отправлять письмо в хайлайтер, не преобразовывать к html
        ShowContentMeta - посылать в томиту запрос на экстракцию фактов
        CutSignature - отрезать от письма подпись
        XmlStreamerOn - вытащить cid-ы из текста письма
        sms_hash - хеш, который приходит в урле для смс-писем. Нужен для оборачивания ссылок в смс-письмах
        tz - паспортная тайм-зона юзера. Если нет параметра, берется дефолтная "Europe/Moscow"
        user_domain - домен, для ПДД. Нужен для оборачивания ссылок
        auth_domain - домен, с которого пришел запрос (.yandex.ru, .yandex.ua, .yandex-team.ru итд). Нужен для оборачивания ссылок
        pdd - 1 для ПДД, для остальных пустой. Нужен для оборачивания ссылок
        sanitize_level - spam для спамовых писем, для остальных пустой. Нужен для санитайзера
        secure - 1 если к wmi пришел https-запрос, для остальных пустой. Нужен для санитайзера и оборачивания ссылок
        """

        api_name = 'message'
        assert 'stid' in kw
        resp = self.http.get(self.url + '/' + api_name, params=kw)

        if self.dry_run:
            return

        return self.check_and_parse(resp)


class Message:
    """  Класс представляет одно сообщение, полученное от message_body """

    def __init__(self, data, mid=None, subtype=None, maillist=None):
        self.data = data
        self.mid = mid
        self.subtype = subtype
        self.maillist = maillist
        self.is_translated = False

    def get_mentioned_emails(self):
        for d in self.data['info']['addressesResult']:
            if d['direction'] in ['to', 'cc']:
                yield d['email']

    @property
    def stid(self):
        return self.data['info']['stid']

    def bodies(self):
        return [MessagePart(d, subtype=self.subtype) for d in self.data['bodies']]

    get_parts = bodies

    def get_attachments(self):
        return [Attachment(d) for d in self.data['attachments']]


class MessagePart:
    """ Один part сообщения """

    def __init__(self, data, subtype=None):
        self.data = data
        self.subtype = None

    @property
    def subtype(self):
        if self.subtype is None:
            return self.data[u'transformerResult']['textTransformerResult']['typeInfo']['contentType']['subtype']
        else:
            return self.subtype

    @property
    def content(self):
        return self.data[u'transformerResult']['textTransformerResult']['content']

    @property
    def content_id(self):
        return self.data['transformerResult']['textTransformerResult']['typeInfo']['contentId']

    @property
    def hid(self):
        return self.data[u'hid']


class Attachment:
    """ Один аттачмент сообщения """

    def __init__(self, data):
        self.data = data
        self._transformer_key = None
        for key in self.data.keys():
            if key.endswith('TransformerResult'):
                self._transformer_key = key
                break

    @property
    def content(self):
        return NotImplemented

    @property
    def hid(self):
        return self.data[self._transformer_key]['hid'] if self._transformer_key else u''

    @property
    def disposition_filename(self):
        if self._transformer_key:
            ti = self.data[self._transformer_key]['typeInfo']
            return ti.get('dispositionFilename') or ti.get('name') or u''
        return u''

    @property
    def length(self):
        return self.data[self._transformer_key]['typeInfo']['length'] if self._transformer_key else u''

    @property
    def content_type(self):
        sect = self._get_content_type_section()
        if sect:
            return sect['type']
        return u''

    @property
    def content_subtype(self):
        sect = self._get_content_type_section()
        if sect:
            return sect['subtype']
        return u''

    @property
    def is_inline(self):
        return self.data[self._transformer_key].get('isInline', False) if self._transformer_key else False

    @property
    def content_id(self):
        return self.data[self._transformer_key].get('cid') if self._transformer_key else u''

    def _get_content_type_section(self):
        transf_section = self.data[self._transformer_key] if self._transformer_key else {}
        if 'contentType' in transf_section:
            return transf_section['contentType']
        elif 'contentType' in transf_section.get('typeInfo', {}):
            return transf_section['typeInfo']['contentType']
        return None
