# coding: utf-8
from .compat import compat_requests_session, compat_requests_json
from django.conf import settings

from base import BaseMailInteractionBackend, wrap_session


class PDDError(Exception):
    def __init__(self, message, code=None):
        self.message = message
        self.code = code
        super(PDDError, self).__init__(message, code)


class PDDBackend(BaseMailInteractionBackend):

    backend_type = 'pdd'

    def __init__(self,
                 url=settings.PDD_API_URL,
                 timeout=settings.PDD_API_TIMEOUT,
                 dry_run=False,
                 logger=None, **kwargs):
        super(PDDBackend, self).__init__()

        if logger:
            self.log = logger

        session = compat_requests_session(timeout=timeout, **kwargs)
        self.http = wrap_session(session, dry_run=dry_run, logger=logger)
        self.url = url
        self.dry_run = dry_run

    def check_and_parse(self, response, ignore_errors=None):
        try:
            self.error_test(response)
        except IOError:
            if response.status_code == 400:
                raise PDDError("Bad request. %s" % response.url)
            else:
                raise

        json = compat_requests_json(response)
        if json.get('success') == 'error':
            error = json.get('error')
            if ignore_errors and error in ignore_errors:
                pass
            else:
                raise PDDError(message='PDDError', code=json.get('error'))
        return json

    def register_domain_for_corp_maillists(self, domain):
        """
        Подключить домен для корпоративных рассылок
        См. http://pdd-back01g-dev.cmail.yandex.net/iapi/domain/register_domain_for_corp_maillists?help=internal
        и таск ML-1466

        domain: домен, например, 'money.yandex.ru'
        """

        resp = self.http.post(self.url + 'domain/register_domain_for_corp_maillists',
                              data={'uid': settings.PDD_DOMAIN_ADMIN_UID, 'domain': domain, 'ip': '127.0.0.1'},
                              headers=settings.PDD_API_AUTH_HEADER)
        return self.check_and_parse(resp, ignore_errors=['already_exists',])

    def create_corp(self, domain, maillist, subscriber):
        """
        Прописать рассылку в ПДД
        См. http://pdd-back01g-dev.cmail.yandex.net/iapi/email/ml/create_corp?help=internal
        и таск ML-1466

        maillist: логин рассылки, строка
        domain: домен, строка
        subscriber: служебный адрес для пересылки почты, строка
        """

        if '@' in maillist:
            maillist = maillist.split('@', 1)[0]

        resp = self.http.post(self.url + 'email/ml/create_corp',
                              data={'uid': settings.PDD_DOMAIN_ADMIN_UID,
                                    'domain': domain,
                                    'maillist': maillist,
                                    'subscriber': subscriber,
                                    'ip': '127.0.0.1'},
                              headers=settings.PDD_API_AUTH_HEADER)
        return self.check_and_parse(resp, ignore_errors=['occupied',])
