from django.conf import settings

from mlcore.interaction.base import BaseMailInteractionBackend, wrap_session
from mlcore.interaction.compat import compat_requests_session


class StorageError(Exception):
    def __init__(self, message, code=None):
        self.message = message
        self.code = code
        super(StorageError, self).__init__(message, code)

class NotFoundStorageError(StorageError):
    pass


class StorageBackend(BaseMailInteractionBackend):
    backend_type = 'storage'

    def __init__(self, url=settings.STORAGE_API_URL, timeout=20,
                 dry_run=False, logger=None, **kwargs):
        super(StorageBackend, self).__init__()

        if logger:
            self.log = logger

        session = compat_requests_session(timeout=timeout, **kwargs)
        self.http = wrap_session(session, dry_run=dry_run, logger=logger)
        self.url = url
        self.dry_run = dry_run
        self.params = {'service': 'ml'}

    def check_and_parse(self, response):
        try:
            self.error_test(response)
        except IOError:
            if response.status_code == 400:
                raise StorageError("Bad request. %s" % response.url)
            elif response.status_code == 404:
                raise NotFoundStorageError('Message not found in storage')
            else:
                raise
        return response.content

    def get(self, stid, offset_begin=None, offset_end=None):
        range_header = None
        if offset_begin is not None and offset_end is not None:
            range_header = {'Range': "bytes={begin}-{end}".format(begin=offset_begin, end=offset_end)}
        resp = self.http.get(self.url + '/gate/get/' + stid,
                             headers=range_header,
                             params=self.params)

        return self.check_and_parse(resp)
