# coding: utf-8

# Удаление письма из рассылки через API webcorp (ML-1011)
# https://wiki.yandex-team.ru/users/shelkovin/asyncoperations/http-interface

from base import BaseMailInteractionBackend, wrap_session
from django.conf import settings
from mlcore.utils import blackbox_util
from .compat import compat_requests_session, compat_requests_json


class YWebCorpError(Exception):
    def __init__(self, message, code=None):
        self.message = message
        self.code = code
        super(YWebCorpError, self).__init__(message, code)


class YWebCorpBackend(BaseMailInteractionBackend):
    backend_type = 'webcorp'

    def __init__(self, url=settings.WEBCORP_URL, timeout=settings.WEBCORP_URL_TIMEOUT,
                 dry_run=False, logger=None, destination_id=settings.WEBCORP_TVM_ID, **kwargs):
        super(YWebCorpBackend, self).__init__()

        if logger:
            self.log = logger

        session = compat_requests_session(timeout=timeout, **kwargs)
        self.http = wrap_session(session, dry_run=dry_run, logger=logger, destination_id=destination_id)
        self.url = url
        self.dry_run = dry_run

    def check_and_parse(self, response):
        """
        В случае успеха возвращает result == ok
        """
        json = compat_requests_json(response)
        if json.get('result') == 'ok':
            return json
        else:
            raise YWebCorpError("Internal error: %s" % json.get('error'))

    def _required_params(self, login):
        """
        Общие для всех запросов аргументы (кроме /ping):
         - uid
         - suid
         - mdb
         - login
         - karma
         - karma_status
         - ora_dsn - deprecated - всегда "mail_dsn"
        """
        bdata = blackbox_util.get_userinfo_by_login(login)
        data = {
            'ora_dsn': 'mail_dsn',
            'mdb': settings.WEBCORP_MDB,
            'login': login,
            'uid': bdata.get('uid'),
            'suid': bdata.get('fields', {}).get('suid'),
            'karma': bdata.get('karma'),
            'karma_status': bdata.get('karma_status') or bdata.get('karma'),
        }
        return data

    def remove(self, login, tids=None, mids=None):
        """
        Смотрит на каждое письмо, если оно лежит в корзине - удаляет его,
        если нет - перемещает в корзину, отправляя репорт в постмастер.
        Если передаем tids, то удаляются все перечисленные треды.
        Если только mids, то удаляется одно письмо.
        """
        api_name = 'remove'
        data = self._required_params(login)
        if tids:
            data.update({
                'tids': tids,
            })
        elif mids:
            data.update({
                'mids': mids,
            })
        else:
            raise YWebCorpError("Required mids or tids, but not together.")

        resp = self.http.post(self.url+'/'+api_name, data=data)
        return self.check_and_parse(resp)