# coding: utf-8

"""
Имплементация апи фуриты для автоответов https://wiki.yandex-team.ru/Users/mio369/yfurita
"""

from base import BaseMailInteractionBackend, wrap_session
from django.conf import settings

from .compat import compat_requests_session, compat_requests_json
from mlcore.utils.getters import get_list_uid
from mlcore.utils.blackbox_util import get_userinfo
from mlcore.ml.models import CorporateDomain


class YFuritaError(Exception):
    def __init__(self, message, code=None):
        self.message = message
        self.code = code
        super(YFuritaError, self).__init__(message, code)


class YFuritaBackend(BaseMailInteractionBackend):
    backend_type = 'yfurita'

    def __init__(self, url=settings.FURITA_URL, tvm_id=settings.FURITA_TVM_ID,
                 timeout=settings.FURITA_TIMEOUT, dry_run=False, logger=None, is_yateam=True, **kwargs):
        super(YFuritaBackend, self).__init__()

        if logger:
            self.log = logger

        session = compat_requests_session(timeout=timeout, **kwargs)
        self.http = wrap_session(session, dry_run=dry_run, logger=logger)
        self.url = url
        self.tvm_id = tvm_id
        self.dry_run = dry_run
        self.is_yateam = is_yateam

    def check_and_parse(self, response, verify_status=True):
        if response.status_code == 404:
            raise YFuritaError("Bad request. %s" % response.url, response.status_code)

        json = compat_requests_json(response)
        if verify_status and json.get('status') == 'error':
            raise YFuritaError("Error: %s" % json.get('report'), response.status_code)
        return json

    def _request(self, method, uid, mdb, **kw):
        kw['uid'] = uid
        kw['db'] = mdb
        if self.is_yateam:
            settings_key = 'team'
        else:
            settings_key = 'other'
        furita_url = self.url[settings_key]
        furita_tvm_id = self.tvm_id[settings_key]
        req_url = furita_url
        url = "%s/api/%s.json" % (req_url, method)
        # sic: wrapped session pops destination_id kwarg if present
        r = self.http.get(url, params=kw, destination_id=furita_tvm_id)
        return self.check_and_parse(r)

    def list(self, uid, mdb=None, id=None, detailed=None):
        params = {}
        if id is not None:
            params['id'] = id
        if detailed is not None:
            params['detailed'] = detailed
        return self._request('list', uid=uid, mdb=mdb, **params)

    def remove(self, uid, id, mdb=None):
        return self._request('remove', uid=uid, mdb=mdb, id=id)

    def edit(self, id, rule, uid, mdb=None):
        params = rule.copy()
        if id is not None:
            params['id'] = id
        return self._request('edit', uid=uid, mdb=mdb, **params)

    def create(self, rule, uid, mdb=None):
        return self.edit(uid=uid, mdb=mdb, id=None, rule=rule)

    def order(self, uid, ids, mdb=None):
        return self._request('order', mdb=mdb, uid=uid, list=ids)

    def enable(self, uid, a_id, enabled, mdb=None):
        return self._request('enable', uid=uid, mdb=mdb, id=a_id, enabled=enabled)


class BaseRule(object):
    """
    Базовый класс для правил ящика
    """

    def __init__(self, dry_run=False):
        self.yfurita = YFuritaBackend(dry_run=dry_run)
        self.mdb = settings.FURITA_MDB

    def _get_bb_data(self, maillist):
        """ Получение uid и mdb для списка с учетом типа домена """
        if not maillist.email or '@' not in maillist.email:
            return get_list_uid(maillist), self.mdb, True
        list_domain = maillist.email.split('@')[-1]
        domain = CorporateDomain.objects.get(domain=list_domain)
        if domain.is_pdd:
            uinfo = get_userinfo(maillist.email, is_yateam=False)
            return uinfo.get('uid', None), uinfo.get('fields', {}).get('mdb', ''), False

        return get_list_uid(maillist), self.mdb, True

    def isenable(self, maillist, rule_id):
        """ Доступно ли правило """
        uid, mdb, self.yfurita.is_yateam = self._get_bb_data(maillist)
        rules = self.yfurita.list(uid=uid, id=rule_id, mdb=mdb)['rules']
        return True if rules[0]['enabled'] else False

    def remove(self, maillist, rule_id):
        """ Удаляем правило автоответа по заданному id и maillist
        Например:
        curl -i 'https://furita.mail.yandex-team.ru:443/api/remove.json?user=1120000000145218&db=mdb100&id=2370000000000096533'
        """
        uid, mdb, self.yfurita.is_yateam = self._get_bb_data(maillist)
        self.yfurita.remove(uid, rule_id, mdb=mdb)

    def order(self, maillist, ids):
        """
        Изменяем приоритеты правил для автоответов:
        curl -i 'https://furita.mail.yandex-team.ru:443/api/order.json?db=mdb100&user=1120000000145218&list=2370000000000098430,2370000000000096874,2370000000000098429'
        """
        uid, mdb, self.yfurita.is_yateam = self._get_bb_data(maillist)
        return self.yfurita.order(uid, ids, mdb=mdb)

    def enable(self, maillist, a_id, enabled):
        """
        Включаем или выключаем автоответ по id:
        curl -i 'https://furita.mail.yandex-team.ru:443/api/enable.json?db=mdb100&user=1120000000145218&id=2370000000000098430&enabled=0'
        """
        uid, mdb, self.yfurita.is_yateam = self._get_bb_data(maillist)
        return self.yfurita.enable(uid, a_id, enabled, mdb=mdb)


class Autoreply(BaseRule):
    """
    Все примеры для рассылки: testdomainml@yandex-team.com
    """

    RULE_NAME = '_ML_AUTOREPLY_'

    def create(self, maillist, autoanswer, field1='', field2='', field3=''):
        """ Добавляем новое правило автоответа:
        Например:
        $ curl -i 'https://furita.mail.yandex-team.ru:443/api/edit.json?db=mdb100&user=1120000000145218&clicker=reply&autoanswer=hello,vasya&stop=1&noconfirm=1&lang=ru&field1=cc&field2=4&field3=fantasy&name=_ML_AUTOREPLY_'
        """
        uid, mdb, self.yfurita.is_yateam = self._get_bb_data(maillist)

        if not int(field2) in (1, 2, 3, 4):
            raise Exception("Incorrect field2")

        rule = {
            'clicker': 'reply',
            'autoanswer': autoanswer,
            'stop': 0,
            'noconfirm': 1,
            'field1': field1,
            'field2': field2,
            'field3': field3,
            'name': self.RULE_NAME,
        }
        frule = self.yfurita.create(uid=uid, rule=rule, mdb=mdb)
        return frule['id']

    def detailed(self, maillist):
        """
        Возвращаем всю информацию о правилах, начинающихся с _ML_AUTOREPLY_.
        Например:
        list(
            dict(id=1, msg='hello',         field1='from', field2=3, field3='ya.ru'),
            dict(id=2, msg='hello, vasya',  field1='cc', field2=4, field3='fantasy@'),
            ...
        )
        Запрос:
        curl -s 'https://furita.mail.yandex-team.ru:443/api/list.json?user=1120000000145218&db=mdb100&detailed=1' | python -mjson.tool
        """

        uid, mdb, self.yfurita.is_yateam = self._get_bb_data(maillist)

        autoreply_rules = []
        all_rules = self.yfurita.list(uid=uid, detailed=1, mdb=mdb)['rules']
        for rule in all_rules:
            if rule['name'].startswith(self.RULE_NAME):
                a_rule = dict()
                a_rule['id'] = rule['id']
                a_rule['enabled'] = rule['enabled']
                a_rule['priority'] = rule['priority']
                action = rule['actions'][0]
                if action['type'] == 'reply':
                    a_rule['msg'] = action['parameter']

                conditions = rule["conditions"]
                for condition in conditions:
                    if len(conditions) > 1 and condition.get('div', '') == 'nospam':
                        continue
                    a_rule['field1'] = condition.get('div', '')
                    a_rule['field2'] = condition.get('oper', '')
                    a_rule['field3'] = condition.get('pattern', 'nopattern')
                    autoreply_rules.append(a_rule)

        return sorted(autoreply_rules, key=lambda a: a['priority'], reverse=True)


class Forward(BaseRule):
    """
    Класс управления форвардом для внешних ящиков
    """
    RULE_PREFIX = '_ML_DIRECT_FORWARD_'

    @classmethod
    def rule_name(cls, forward_to):
        rule_email = forward_to.replace('.', '-').replace('@', '-at-').upper()
        return '{}{}'.format(cls.RULE_PREFIX, rule_email)

    def create(self, maillist, forward_to):
        """
        Добавляем новое правило форвардинга
        """
        raise Exception('Furita forwards forbidden')

    def detailed(self, maillist):
        """
        Возвращаем всю информацию о правилах, начинающихся с _ML_DIRECT_FORWARD_.
        """

        uid, mdb, self.yfurita.is_yateam = self._get_bb_data(maillist)

        _rules = []
        all_rules = self.yfurita.list(uid=uid, detailed=1, mdb=mdb)['rules']
        for rule in all_rules:
            if rule['name'].startswith(self.RULE_PREFIX):
                _rule = dict()
                _rule['id'] = rule['id']
                _rule['name'] = rule['name']
                _rule['enabled'] = rule['enabled']
                _rule['priority'] = rule['priority']
                action = rule['actions'][0]
                if action['type'] == 'forwardwithstore':
                    _rule['to'] = action['parameter']
                _rules.append(_rule)

        return sorted(_rules, key=lambda a: a['priority'], reverse=True)


class SpecialMailListFolder(object):
    """
    Обработка правил Furit'ы для рассылок в "специальной" папке
    """

    RULE_PREFIX = '_ML_CUSTOM_FOLDER_SUBSCRIBE_'

    def __init__(self, dry_run=False, mdb=settings.FURITA_MDB):
        self.yfurita = YFuritaBackend(dry_run=dry_run)
        self.mdb = mdb

    def __get_rule_name(self, maillist):
        return '{0}_{1}_'.format(
            self.RULE_PREFIX,
            maillist.pk
        )

    def find_rule(self, maillist, user, folder_id=None):
        """
        Проверяем наличие правила для перенаправления писем из Входящих в папку
        """
        uid = user.staff.uid
        all_rules = self.yfurita.list(uid=uid, detailed=1, mdb=self.mdb)['rules']
        rule_name = self.__get_rule_name(maillist)
        for rule in all_rules:
            if rule['name'] == rule_name and (rule.get('move_folder') == folder_id or folder_id is None):
                return rule['id']

        return False

    def create_rule(self, maillist, user, folder_id):
        """
        Создаем правило для пересылки из Inbox в специальную папку
        """
        uid = user.staff.uid
        rule = {
            'clicker': 'move',
            'stop': 1,
            'noconfirm': 1,
            'field1': ['to', 'cc'],
            'field2': [3, 3],
            'field3': [maillist.get_email(), maillist.get_email()],
            'move_folder': folder_id,
            'name': self.__get_rule_name(maillist),
        }

        frule = self.yfurita.create(uid=uid, rule=rule, mdb=self.mdb)
        return frule['id']

    def remove_rule(self, maillist, user, rule_id=None):
        """
        Удаляем правило для рассылки
        """
        rule_id = rule_id if rule_id is not None else self.find_rule(maillist, user)
        if rule_id:
            self.yfurita.remove(user.staff.uid, rule_id, mdb=self.mdb)
