# coding: utf-8

"""
Имплементация апи почты http://wiki.yandex-team.ru/pochta/mailyateam/MLApi
"""

from xml.dom import minidom
from xml.parsers.expat import ExpatError
from lxml import etree
from urllib import urlencode

from django.conf import settings

from base import BaseMailInteractionBackend, wrap_session
from mlcore.utils.getters import get_suid, get_list_suid
from .compat import compat_requests_session, compat_requests_json


class YmailError(Exception):
    def __init__(self, message, code=None):
        self.message = message
        self.code = code
        super(YmailError, self).__init__(message, code)


class YmailBackend(BaseMailInteractionBackend):
    backend_type = 'ymail'

    def __init__(self, url=settings.YMAIL_API_URL, timeout=20,
                 dry_run=False, logger=None, **kwargs):
        super(YmailBackend, self).__init__()

        if logger:
            self.log = logger

        session = compat_requests_session(timeout=timeout, **kwargs)
        self.http = wrap_session(session, dry_run=dry_run, logger=logger)
        self.url = url
        self.dry_run = dry_run

    def check_and_parse(self, response, verify_status=True):
        try:
            self.error_test(response)
        except IOError:
            if response.status_code == 400:
                raise YmailError("Bad request. %s" % response.url)
            else:
                raise

        json = compat_requests_json(response)
        if isinstance(json, dict) and verify_status:
            if json.get('status') == 'ok':
                return json
            elif json.get('status') == 'fail':
                raise YmailError(json.get('reason', ''), json.get('code'))
            elif json.get('status') == 'error':
                raise YmailError(json.get('error'), 'error')
            else:
                raise YmailError("Wrong response status %s" % json)

        return json

    def create_maillist(self, maillist, parent=None):
        """ Создает рассылку """
        api_name = 'CreateList'

        data = {'folder_uname': get_list_suid(maillist)}
        if parent is not None:
            data['parent_uname'] = get_list_suid(parent)

        resp = self.http.get(self.url + api_name, params=data)

        if self.dry_run:
            return

        self.check_and_parse(resp)

    def folders(self, maillist):
        """ Возвращает список папок в рассылке
        maillist - suid папки или объект models.MailList, name
            [
                {
                    "fid": 2370000060000036449,
                    "fname": "foo-bar-baz",
                    "path": "Yandex|foo-bar-baz"
                }
            ]
        """
        api_name = 'SharedFids'
        suid = get_list_suid(maillist)
        resp = self.http.get(self.url + api_name, params={'folder_uname': suid})

        if self.dry_run:
            return

        return self.check_and_parse(resp)

    def subscribe(self, user, maillist):
        """ Подписка пользователя на рассылку
        user - suid пользователся, login, Staff или User
        maillist - suid папки или MailList
        """
        api_name = 'Subscribe'
        uname = get_suid(user)
        folder_uname = get_list_suid(maillist)
        resp = self.http.get(self.url + api_name, timeout=10,
                             params={'uname': uname, 'folder_uname': folder_uname})

        if self.dry_run:
            return

        self.check_and_parse(resp)

    def unsubscribe(self, user, maillist):
        """ Отписка """
        api_name = 'UnSubscribe'
        uname = get_suid(user)
        folder_uname = get_list_suid(maillist)
        resp = self.http.get(self.url + api_name, timeout=10,
                             params={'uname': uname, 'folder_uname': folder_uname})

        if self.dry_run:
            return

        self.check_and_parse(resp)

    def subscriptions(self, user, with_zombies=False):
        """ Отдает все подписки пользователя по умолчанию без зомби папок """
        api_name = 'Subscriptions'
        uname = get_suid(user)
        with_zombies = int(with_zombies)
        resp = self.http.get(self.url + api_name,
                             params={'uname': uname, 'with_zombies': with_zombies})

        if self.dry_run:
            return

        return self.check_and_parse(resp)

    def subscribers(self, maillist):
        """
        Отдает suid-ы пользователей, подписанных на общую папку.
        Пример:
        curl -s 'http://mlapi.mail.yandex.net/mdb305/ymail.MLApi.Subscribers?folder_uname=1120000000014074'| python -mjson.tool
        [
            1120000000131872,
            1120000000098076,
            1120000000130352,
            1120000000016367,
            ...
        ]
        """
        api_name = 'Subscribers'
        folder_uname = get_list_suid(maillist)
        resp = self.http.get(self.url + api_name,
                             params={'folder_uname': folder_uname})

        if self.dry_run:
            return

        return self.check_and_parse(resp, verify_status=False)

    def create_passport_user_old_method_api(self, name, email):
        """ Создает пользователя в yateam паспорте """
        PASSPORT_REGISTER_URL = (("%s/passport?mode=admsimplereg"
                                  "&login=%s&maillist=1&external_email=%s")
                                 % (settings.PASSPORT_URL, name, email))
        response = self.http.get(PASSPORT_REGISTER_URL, verify=settings.PASSPORT_SSL_VERIFY)
        if response.status_code != 200:
            raise ValueError(u"Проблемы при создании пользователя %s", name)

        if self.dry_run:
            return

        xml = response.content

        try:
            dom = minidom.parseString(xml)
        except ExpatError, e:
            self.log.exception(u"Expat error on imap user creation: %s", e)
            raise PassportException("Invalid xml response from passport", e)

        if ((dom.documentElement.getAttribute("status") == "error") or
                dom.documentElement.getElementsByTagName("error")):
            self.log.error(u"User creation in passport failed with error: %s", xml)
            raise PassportException("User creation in passport failed with error",
                                    xml)

    def create_passport_user(self, imap_name, email):
        """
        Создание рассылок в паспорте с помощью нового метода: intranet
        """
        url = '%s/1/bundle/account/register/intranet/?consumer=mail' % (settings.PASSPORT_URL)
        login_data = {
            'login': imap_name,
            'is_maillist': True,
            'default_email': email,
            'externalmail_alias': email,
        }
        login, domain = email.split('@')
        if domain in settings.PASSPORT_ALIASES:
            login_data['altdomain_alias'] = email
        response = self.http.post(url, data=login_data, headers={'Ya-Consumer-Client-Ip': '127.0.0.1'},
                                  verify=settings.PASSPORT_SSL_VERIFY)  # TODO: отправлять в паспорт ip клиента?
        if self.dry_run:
            return
        return self.check_and_parse(response, verify_status=False)

    def change_mdb(self, suid, current_mdb, new_mdb):
        url = '%s/mailhost' % settings.PASSPORT_URL
        params = {
            'db_id': new_mdb,
            'suid': suid,
            'old_db_id': current_mdb,
            'op': 'assign'
        }
        response = self.http.get('{}?{}'.format(url, urlencode(params)))
        if response.status_code != 200:
            raise PassportException("Can't change mdb for suid=%s", suid)

        try:
            response_tree = etree.XML(response.content)
            status_id = int(response_tree.find('status').get('id'))
            if status_id != 0:
                raise PassportException("Change mdb failed for suid=%s", suid)
        except (etree.LxmlError, ValueError, AttributeError):
            raise PassportException("Wrong response from passport suid=%s", suid)

    def passport_alias_altdomain(self, suid, email):
        """
        Запрос на создание алиаса ML-1463
        """
        url = '%s/1/account/%s/alias/altdomain/?consumer=mail' % (settings.PASSPORT_URL, suid)
        resp = self.http.post(url, data={'alias': email}, headers={'Ya-Consumer-Client-Ip': '127.0.0.1'},
                              verify=settings.PASSPORT_SSL_VERIFY)
        if self.dry_run:
            return
        return self.check_and_parse(resp, verify_status=False)

    def mails_in_thread(self, thread_id):
        """ Список писем в треде """
        api_name = 'MailsInThread'
        resp = self.http.get(self.url + api_name,
                             params={'thread_id': thread_id})
        return self.check_and_parse(resp, verify_status=False)

    def get_archiving_options(self, maillist):
        """
        Получить статус авторотируемости(удаление писем через n-дней).
        Пример:
        http -b GET 'http://mdbdir.mail.yandex.net/mdb305/ymail.mlapi.GetArchivingOptions?uname=1120000000018516'
        {
            "enabled": false,
            "keep_days": null
        }
        """

        api_name = 'GetArchivingOptions'
        uname = get_suid(maillist)
        resp = self.http.get(self.url + api_name,
                             params={'uname': uname})

        return self.check_and_parse(resp, verify_status=False)

    def set_archiving_options(self, maillist, enable=False, keep_days=None, archiving_type='clean'):
        """
        Установить значения для авторотируемости.
        Пример:
        http -b GET 'http://mdbdir.mail.yandex.net/mdb305/ymail.mlapi.SetArchivingOptions?uname=1120000000018516&enable=1&keep_days=42&type=archive'
        {
            "status": "ok"
        }
        """
        api_name = 'SetArchivingOptions'
        uname = get_suid(maillist)
        if enable and keep_days:
            resp = self.http.get(self.url + api_name,
                                 params={'uname': uname, 'enable': 1,
                                         'keep_days': keep_days,
                                         'type': archiving_type})
        else:
            resp = self.http.get(self.url + api_name,
                                 params={'uname': uname, 'enable': 0})

        return self.check_and_parse(resp)

    def rename_list(self, maillist, changed_name):
        """
        Изменить название рассылки. (Пока не используется)
        Пример:
        http -b GET 'http://mdbdir.mail.yandex.net/mdb305/ymail.mlapi.RenameList?folder_uname=1120000000127504&new_name=mltest_hello_kitty'
        {
            "status": "ok"
        }

        """
        api_name = 'RenameList'
        folder_uname = get_list_suid(maillist)
        resp = self.http.get(self.url + api_name,
                             params={'folder_uname': folder_uname, 'new_name': changed_name})

        return self.check_and_parse(resp)

    def set_parent(self, maillist, parent_maillist):
        """
        Перенос фолдеров по дереву.
        Если parent_maillist == None, то переносим в корень Yandex.
        Пример:
        http -b GET 'http://mlapi.mail.yandex.net/mdb305/ymail.MLApi.MoveList?folder_uname=1120000000127504&new_parent_uname=1120000000121823'
        {
            "status": "ok"
        }
        """

        api_name = 'MoveList'
        folder_uname = get_list_suid(maillist)
        if parent_maillist:
            new_parent_uname = get_list_suid(parent_maillist)
        else:
            new_parent_uname = 0
        resp = self.http.get(self.url + api_name,
                             params={'folder_uname': folder_uname, 'new_parent_uname': new_parent_uname})

        return self.check_and_parse(resp)

    def get_parent(self, maillist):
        """
        Посмотреть родителя по suid'у рассылки.
        Пример:
        http -b GET 'http://mdbdir.mail.yandex.net/mdb305/ymail.mlapi.SharedFolderParent?folder_uname=1120000000097032'
        [
            {
                "fname": "sm",
                "parent_uname": 1120000000097032,
                "uname": 1120000000044833
            }
        ]
        """

        api_name = 'SharedFolderParent'
        folder_uname = get_list_suid(maillist)
        resp = self.http.get(self.url + api_name,
                             params={'folder_uname': folder_uname})

        return self.check_and_parse(resp)

    def get_all_children(self):
        """
        Получаем все рассылки.
        Пример:
        http -b GET 'http://mdbdir.mail.yandex.net/mdb305/ymail.mlapi.SharedFolderChildren?folder_uname=0'
        [
            {
                "fname": "git_commits-maps-projects-mobmaps",
                "parent_uname": 0,
                "uname": 1120000000106988
            },
            {
                "fname": "advq-root",
                "parent_uname": 0,
                "uname": 1120000000042815
            },
            ...,
            # все рассылки
        ]
        """

        api_name = 'SharedFolderChildren'
        resp = self.http.get(self.url + api_name,
                             params={'folder_uname': 0})

        return self.check_and_parse(resp, verify_status=False)

    def get_children(self, folder_uname):
        """
        Посмотреть сыновей по suid'у рассылки.
        Пример:
        http -b GET 'http://mdbdir.mail.yandex.net/mdb305/ymail.mlapi.SharedFolderChildren?folder_uname=0'
        [
            {
                "fname": "git_commits-maps-projects-mobmaps",
                "parent_uname": 0,
                "uname": 1120000000106988
            },
            {
                "fname": "advq-root",
                "parent_uname": 0,
                "uname": 1120000000042815
            },
            ...,
            # все рассылки
        ]
        """

        api_name = 'SharedFolderChildren'
        # folder_uname = get_list_suid(maillist)
        resp = self.http.get(self.url + api_name,
                             params={'folder_uname': folder_uname})

        return self.check_and_parse(resp, verify_status=False)

    def set_shared_flags(self, maillist, state):
        """
        Установить общие флажки.
        Пример:
        http -b GET 'http://mlapi.mail.yandex.net/mdb305/ymail.MLApi.UpdateListType?folder_uname=1120000000127504&make_common=1'
        {
            "status": "ok"
        }
        """

        api_name = 'UpdateListType'
        make_common_value = 1 if state else 0
        folder_uname = get_list_suid(maillist)
        resp = self.http.get(self.url + api_name,
                             params={'folder_uname': folder_uname, 'make_common': make_common_value})

        return self.check_and_parse(resp)

    def get_info(self, maillist):
        """
        Посмотреть всю информацию о рассылке.
        Пример:
        http -b GET 'http://mdbdir.mail.yandex.net/mdb305/ymail.mlapi.SharedFolderInfo?folder_uname=1120000000044833'

        {
            "archiving": {
                "enabled": false,
                "keep_days": null
            },
            "fname": "sm",
            "is_common": false,
            "msg_cnt": 15075,
            "parent_uname": 0,
            "uname": 1120000000044833
        }
        """

        api_name = 'SharedFolderInfo'
        folder_uname = get_list_suid(maillist)
        resp = self.http.get(self.url + api_name,
                             params={'folder_uname': folder_uname})

        return self.check_and_parse(resp, verify_status=False)


class PassportException(Exception):
    pass
