# coding: utf-8

# Установка "Персональных данных" рассылки через API Почтовых Настроек (ML-1230)
# https://wiki.yandex-team.ru/users/prez/yserversettings

from base import BaseMailInteractionBackend, wrap_session
from django.conf import settings

from .compat import compat_requests_session, compat_requests_json
from mlcore.utils.blackbox_util import get_userinfo


class YMailSettingsError(Exception):
    def __init__(self, message, code=None):
        self.message = message
        self.code = code
        super(YMailSettingsError, self).__init__(message, code)


def suid_and_mdb(uid, is_yateam):
    # ML-1501: в settings теперь нужно отправлять suid и mdb
    # делаю быстрый фикс, потому что ml редко использует settings
    # для оптимизации нужно добавить поле mdb в модель SuidLookup и хранить там
    ui = get_userinfo(uid_or_email=uid, by_login=False, is_yateam=is_yateam)
    return {'suid': ui['fields']['suid'], 'mdb': ui['fields']['mdb']}


class YMailSettings(BaseMailInteractionBackend):
    backend_type = 'ymailsettings'

    def __init__(self, is_yateam=True, url=settings.YMAIL_SETTINGS_URL, timeout=settings.YMAIL_SETTINGS_TIMEOUT,
                 dry_run=True, logger=None, **kwargs):
        super(YMailSettings, self).__init__()

        if logger:
            self.log = logger

        session = compat_requests_session(timeout=timeout, **kwargs)

        destination_id = settings.YMAIL_SETTINGS_TVM_ID['team' if is_yateam else 'other']
        self.http = wrap_session(session, dry_run=dry_run, logger=logger, destination_id=destination_id)
        self.url = url['team' if is_yateam else 'other']
        self.is_yateam = is_yateam
        self.dry_run = dry_run

    def check_and_parse(self, response, verify_status=False):
        """
        Обязательные для всех запросов параметры:
        suid - идентификатор пользователя в почте
        mdb - имя оракловой базы, на которой живет пользователь
            либо
        uid - идентификатор пользователя в паспорте
        """
        if response.status_code == 404:
            raise YMailSettingsError("Bad request. %s" % response.url, response.status_code)

        json = compat_requests_json(response)
        if verify_status and json.get('status') == 'error':
            raise YMailSettingsError("Error: %s" % json.get('report'), response.status_code)
        return json

    def update_profile(self, uid, from_name):
        """
        Обновляет заданные настройки у пользователя
        Параметры запроса:
        <param_name> - набор параметров, для которых имя совпадает с именем настройки;
        каждый параметр содержит обновлённое значение.
        """
        api_name = 'update_profile'
        data = {
            "uid": uid,
            "from_name": from_name
        }
        data.update(suid_and_mdb(uid, self.is_yateam))
        self.http.post(self.url + '/' + api_name, data=data)

    def get_profile(self, uid, settings_list):
        """
        Возвращает список запрошенных настроек пользователя.
        settings_list - CSV с именами настроек, которые необходимо вернуть.
        В качестве разделителя используется символ '\r' (он же CR, он же %0d)
        """
        api_name = 'get_profile'
        params = {
            "uid": uid,
            "settings_list": settings_list,
            "format": "json"
        }
        params.update(suid_and_mdb(uid, self.is_yateam))
        resp = self.http.get(self.url + '/' + api_name, params=params)
        return self.check_and_parse(resp)

    def get_profile_from_name(self, uid):
        """
        Возвращаем только поле from_name
        """
        resp = self.get_profile(uid, 'from_name')
        return resp['settings']['single_settings']['from_name']
