# encoding: utf-8
from __future__ import unicode_literals

import json

from django.conf import settings

from mlcore.utils.getters import get_list_uid, get_user_uid
from .base import BaseMailInteractionBackend, wrap_session
from .compat import compat_requests_session


class YorkError(Exception):
    pass


class YorkRepeatableError(YorkError):
    # Исклюцение после которому стоит повторить запрос
    pass


class YorkWrongRequestError(YorkError):
    # Неверный формат запроса, не ретраим
    pass


class YorkErrorWithNotification(YorkError):
    # Ошибка, при которой надо отправить письмо пользователю (ML-1790)
    pass


class YorkFatalError(YorkError):
    # Что-то очень плохое случилось, надо фиксить руками, желательно слать нотификации админам
    pass


class YorkBackend(BaseMailInteractionBackend):
    backend_type = 'york'

    def __init__(self, url=settings.YORK_API_URL, timeout=20,
                 dry_run=False, logger=None, destination_id=settings.YORK_TVM_ID, **kwargs):
        super(YorkBackend, self).__init__()

        if logger:
            self.log = logger

        session = compat_requests_session(timeout=timeout, **kwargs)
        self.http = wrap_session(session, dry_run=dry_run, logger=logger, destination_id=destination_id)
        self.url = url
        self.dry_run = dry_run

    @staticmethod
    def _check_response(response):
        if response.status_code == 200:
            return
        elif response.status_code >= 500:
            raise YorkRepeatableError('Something wrong: %s', response.content)
        elif 400 <= response.status_code < 500:
            if settings.YORK_DUPLICATE_FOLDER_MESSAGE in response.json()['error']:
                raise YorkErrorWithNotification
            else:
                raise YorkWrongRequestError('Bad request: %s', response.content)

    def _get_maillist_uid(self, maillist):
        maillist_uid = get_list_uid(maillist)

        if not maillist_uid:
            raise YorkRepeatableError("We couldn't resolve uids maillist_uid=%s", maillist_uid)

        return maillist_uid

    def _get_subscriber_uid(self, user):
        subscriber_uid = get_user_uid(user)
        if not subscriber_uid:
            raise YorkRepeatableError("We couldn't resolve uid ssubscriber_uid=%s", subscriber_uid)

        return subscriber_uid

    def _make_request(self, method, params):
        url = "{}/{}".format(self.url, method)
        response = self.http.get(url, params=params)

        self._check_response(response)
        return response.json()

    def create(self, maillist):
        """
        Создать общую папку
        """
        maillist_uid = self._get_maillist_uid(maillist)
        response = self._make_request('create_list', {
            'uid': maillist_uid
        })
        return int(response['shared_folder_fid'])

    def subscribe(self, user, maillist):
        """
        Подписать пользователя на рассылку
        """
        maillist_uid = self._get_maillist_uid(maillist)
        subscriber_uid = self._get_subscriber_uid(user)

        self._make_request('subscribe', {
            'owner_uid': maillist_uid,
            'subscriber_uid': subscriber_uid,
            'destination_folder_path': json.dumps(maillist.folder_path)
        })

    def unsubscribe(self, user, maillist):
        """
        Отписать пользователя от рассылки
        """
        maillist_uid = self._get_maillist_uid(maillist)
        subscriber_uid = self._get_subscriber_uid(user)

        self._make_request('unsubscribe', {
            'owner_uid': maillist_uid,
            'subscriber_uid': subscriber_uid
        })

    def set_archivation_rule(self, maillist, archivation_type, shared_folder_fid=None, keep_days=None, max_size=None):
        """
        Создает у указанной рассылки правило архивации для шаренной папки
        @param maillist: рассылка
        @param archivation_type: тип архивации ('clean' - письма удаляются, 'archive' - письма раскладываются по годам)
        @param shared_folder_fid: fid шаренной папки, для которой устанавливается правило
        @param keep_days: число дней, за которые нужно оставлять письма (> 0).
        @param max_size: максимальное количество писем, которое нужно оставлять в папке (> 0)
        @return: True, если правило создано/обновлено
        """

        maillist_uid = self._get_maillist_uid(maillist)
        response = self._make_request('set_archivation_rule', {
            'uid': maillist_uid,
            'shared_folder_fid': shared_folder_fid,
            'type': archivation_type,
            'keep_days': keep_days,
            'max_size': max_size
        })
        return response['ok']

    def remove_archivation_rule(self, maillist, shared_folder_fid=None):
        """
        Удаляет у указанного владельца правило для шаренной папки
        @param maillist: рассылка
        @param shared_folder_fid: fid шаренной папки, для которой устанавливается правило
        @return: True, если правило удалено
        """

        maillist_uid = self._get_maillist_uid(maillist)
        response = self._make_request('remove_archivation_rule', {
            'uid': maillist_uid,
            'shared_folder_fid': shared_folder_fid
        })
        return response['ok']
