# coding: utf-8

"""
Имплементация нового апи wmi
ML-1104
https://wiki.yandex-team.ru/users/jkennedy/ywmiapi#sharedfolderssubscribe
https://wiki.yandex-team.ru/users/jkennedy/ywmiapi#sharedfoldersunsubscribe
"""
import requests

from .compat import compat_requests_session, compat_requests_json
from django.conf import settings

from base import BaseMailInteractionBackend, wrap_session
from mlcore.utils.getters import get_suid, get_list_suid, get_user_uid


class YWMIError(Exception):
    def __init__(self, message, code=None):
        self.message = message
        self.code = code
        super(YWMIError, self).__init__(message, code)


class YWMIBackend(BaseMailInteractionBackend):
    backend_type = 'ywmi'

    def __init__(self, url=settings.YWMI_API_URL, mdb=settings.YWMI_API_MDB,
                 oracle_mid_map_url=settings.ORACLE_MID_MAP_API_URL, timeout=20,
                 dry_run=False, logger=None, destination_id=settings.YWMI_TVM_ID, **kwargs):
        super(YWMIBackend, self).__init__()

        if logger:
            self.log = logger

        session = compat_requests_session(timeout=timeout, **kwargs)
        self.http = wrap_session(session, dry_run=dry_run, logger=logger, destination_id=destination_id)
        self.url = url
        self.oracle_mid_map_url = oracle_mid_map_url
        self.mdb = mdb
        self.dry_run = dry_run

    def check_and_parse(self, response):
        try:
            self.error_test(response)
        except IOError:
            if response.status_code == 400:
                raise YWMIError("Bad request. %s" % response.url)
            else:
                raise
        json = compat_requests_json(response)
        if isinstance(json, dict) and 'error' in json:
            raise YWMIError("Bad request. %s" % response.url)
        return json

    def subscribe(self, user, maillist):
        """ Подписка пользователя на рассылку
        user - suid пользователся, login, Staff или User
        maillist - suid папки или MailList
        """

        """
        ywmi:
        shared_folders_subscribe
        Служит для подписки пользователя на рассылку (создание шаренных папок).
        Принимает хттп-аргументы
        suid, mdb, shared_folders_suid:
        suid (обязательный) - suid подписываемого пользователя;
        shared_folders_suid (обязательный) - suid подписки;
        Пример:
         wget -q -O - --header "X-Real-Ip: 37.140.175.107" 'shared_folders_subscribe?suid=1120000000105972&mdb=mdb305&shared_folders_suid=1120000000045509' && echo
        В случае успеха возвращает список шаренных папок, которые были созданы в результате подписки и http код 200.
        Список папок возвращается как рефлекция ywmi::FolderSet, см folders_ex.
        В случае ошибки возвращает рефлекцию ошибки и http код 500.

        """
        api_name = 'shared_folders_subscribe'
        suid = get_suid(user)
        folder_suid = get_list_suid(maillist)
        resp = self.http.get(self.url + '/' + api_name,
                             params={'suid': suid,
                                     'mdb': self.mdb,
                                     'shared_folders_suid': folder_suid})

        if self.dry_run:
            return

        return self.check_and_parse(resp)

    def unsubscribe(self, user, maillist):
        """ Отписка """

        """
        shared_folders_unsubscribe
        Служит для отписки пользователя от рассылки (удаление шаренных папок).
        Принимает хттп-аргументами
        suid, mdb, shared_folders_suid:
        suid (обязательный) - suid пользователя;
        shared_folders_suid (обязательный) - suid подписки, от которой отписываем;
        Пример:
         wget -q -O - --header "X-Real-Ip: 37.140.175.107" 'shared_folders_unsubscribe?suid=1120000000105972&mdb=mdb305&shared_folders_suid=1120000000045509' && echo
        В случае успеха возвращает список шаренных папок, которые были удалены и http код 200.
        Список папок возвращается как рефлекция ywmi::FolderSet, см folders_ex.
        В случае ошибки возвращает рефлекцию ошибки и http код 500.
        """

        api_name = 'shared_folders_unsubscribe'
        suid = get_suid(user)
        folder_suid = get_list_suid(maillist)
        resp = self.http.get(self.url + '/' + api_name,
                             params={'suid': suid,
                                     'mdb': self.mdb,
                                     'shared_folders_suid': folder_suid})

        if self.dry_run:
            return

        return self.check_and_parse(resp)

    def folders(self, user):
        """
        Список папок пользователя
        """
        api_name = 'folders'
        uid = get_user_uid(user)

        resp = self.http.get(self.url + '/' + api_name,
                             params={'uid': uid,
                                     'mdb': self.mdb})

        if self.dry_run:
            return {}

        return self.check_and_parse(resp)['folders']

    def mid_to_stid(self, fuid, mid):
        """ mid письма преобразуем в st_id """
        api_name = 'filter_search'

        resp = self.http.get(self.url + '/' + api_name,
                             params={'uid': fuid,
                                     'folder_set': 'default',
                                     'mids': mid, })
        resp_data = self.check_and_parse(resp)
        if not resp_data:
            return None

        return resp_data.get('envelopes', (None,))[0]

    def mails_in_thread(self, fuid, thread_id):
        """ Список писем в треде """
        api_name = 'messages_by_thread'
        # redirects to hound
        resp = self.http.get(self.oracle_mid_map_url + '/' + api_name,
                             params={'uid': fuid,
                                     'first': 0,
                                     'count': 1024,
                                     'sort_type': 'date_ascending',
                                     'tid': thread_id})
        resp_data = self.check_and_parse(resp)
        if not resp_data:
            return None

        return resp_data.get('envelopes')

    def threads_list(self, fuid, maillist, page_num=1, page_size=3, select_one_next=False):
        """ Список тредов в папке """
        api_name = 'threads_by_folder'
        first = page_num * page_size

        resp = self.http.get(self.url + '/' + api_name,
                             params={'uid': fuid,
                                     'fid': maillist.fid,
                                     'first': first,
                                     'count': page_size + (1 if select_one_next else 0),
                                     },
                             )
        if self.dry_run:
            return None
        resp_data = self.check_and_parse(resp)
        if not resp_data or 'threads_by_folder' not in resp_data:
            return None

        return resp_data['threads_by_folder'].get('envelopes')

    def get_uid_from_mapping(self, thread_id):
        api_name = 'uid_by_mid'
        resp = self.http.get(self.oracle_mid_map_url + '/' + api_name,
                             params={'mid': thread_id})
        if self.dry_run:
            return None
        if resp.status_code == 404:
            raise YWMIError('Thread id=%s is too old and could not be found' % thread_id)
        resp_data = self.check_and_parse(resp)
        if not resp_data:
            return None

        return resp_data.get('uid')

    def mimes(self, uid, mid):
        """
        https://wiki.yandex-team.ru/users/jkennedy/ywmiapi/#mimes
        Возвращает метаинформацию о частях письма (MIME-parts).
        Используется, чтобы получить оффсеты партов, с которыми можно сходить в mds для получения аттачей
        """
        api_name = 'mimes'
        resp = self.http.get(self.url + '/' + api_name,
                             params={'uid': uid,
                                     'mid': mid})
        if self.dry_run:
            return None
        resp_data = self.check_and_parse(resp)
        if not resp_data or 'mimes' not in resp_data or str(mid) not in resp_data.get('mimes'):
            raise YWMIError('Could not get mime-parts for uid=%s, mid=%s' % (uid, mid))

        return resp_data.get('mimes').get(str(mid)).get('root')
