# -*- coding: utf-8 -*-

from django.core.urlresolvers import reverse
from django.shortcuts import get_object_or_404, render_to_response
from django.utils.decorators import method_decorator
from django.contrib.auth.decorators import login_required
from django.template import RequestContext

from django_intranet_stuff.models import Staff
from django_intranet_stuff.utils import translate

from mlcore.ymail.models import get_threads
from mlcore.ml.class_views.base import BaseView
from mlcore.ml.models import MailList, ML_LANGUAGE
from mlcore.mailarchive.translate import translate_thread_list
from mlcore.permissions.utils import can_read


class ThreadList(BaseView):

    @method_decorator(login_required)
    def __call__(self, request, name, language_from=None, language_to=None,
                 language=None, **kwargs):

        self.user = request.user
        self.list = get_object_or_404(MailList, name=name)
        if not can_read(request.user, self.list):
            return self._http_forbidden(request, self.list)
        elif self.list.is_deleted:
            return self._list_marked_as_deleted(request, self.list)

        if self.user.is_authenticated():
            request.session["last_viewed_list"] = self.list.name

        fid = self.list.fid

        if not fid:
            return self._archive_not_supported(request, self.list)

        page_str = request.GET.get("page", 0)
        try:
            page = int(page_str)
        except ValueError:
            page = 0

        thread_list = get_threads(self.list, page_number=page)

        if thread_list:
            from_emails = set(thread_list.get_from_emails())
        else:
            from_emails = set()

        users_email_map = dict(
            (staff.work_email, staff)
            for staff in Staff.objects.filter(work_email__in=from_emails)
        )

        if language_from and language_to:
            language_from = language_from.lower()
            language_to = language_to.lower()
            thread_list = translate_thread_list(thread_list, language_from, language_to)
        elif language in ML_LANGUAGE.items:
            language_from = self.list.preferred_language
            language_to = language
            thread_list = translate_thread_list(thread_list, language_from, language_to)

        if language_to:
            current_translate = '%s-%s' % (language_from, language_to)
        else:
            current_translate = None

        context = {
            "list": self.list,
            "user": request.user,
            "is_owner": self._is_user_owner(),
            "section": "threads",
            "thread_list": thread_list,
            "page": page,
            "users_email_map": users_email_map,
            'language_from': language_from,
            'language_to': language_to,
            'translate_languages': translate.languages,
            'current_translate': current_translate,
            'is_content_translated': language_to is not None,
            'untranslated_url': reverse('mailarchive:thread_list', args=(name,)),
        }
        return render_to_response("mailarchive/thread_list.html",
                                  RequestContext(request, context))
