# -*- coding: utf-8 -*-
import logging

from django.core.urlresolvers import reverse
from django.shortcuts import render_to_response, get_object_or_404
from django.utils.decorators import method_decorator
from django.contrib.auth.decorators import login_required
from django.template import RequestContext

from django_intranet_stuff.models import Staff
from django_intranet_stuff.utils import translate
from mlcore.ymail.models import get_thread
from mlcore.ml.class_views.base import BaseView
from mlcore.ml.models import MailList, ML_LANGUAGE
from mlcore.mailarchive.translate import translate_thread
from mlcore.permissions.utils import can_read

logger = logging.getLogger(__name__)


class ThreadNode(BaseView):

    @method_decorator(login_required)
    def __call__(self, request, name, thread_id,
                 language=None, language_from=None, language_to=None, **kwargs):

        maillist = get_object_or_404(MailList, name=name)
        if not can_read(request.user, maillist):
            return self._http_forbidden(request, maillist)
        elif maillist.is_deleted:
            return self._list_marked_as_deleted(request, maillist)

        fid = maillist.fid

        if not fid:
            return self._archive_not_supported(request, maillist)

        thread = get_thread(maillist, thread_id)

        if thread is None:
            # Ошибка от почты
            return render_to_response('mailarchive/thread_node.html',
                                      {'thread': thread}, RequestContext(request))

        if language_from and language_to:
            language_from = language_from.lower()
            language_to = language_to.lower()
            thread = translate_thread(thread, language_from, language_to)
        elif language in ML_LANGUAGE.items:
            language_from = maillist.preferred_language
            language_to = language
            thread = translate_thread(thread, language_from, language_to)

        if language_to:
            current_translate = '%s-%s' % (language_from, language_to)
        else:
            current_translate = None

        mentioned_emails = set()

        messages = thread.messages
        for m in messages:
            mentioned_emails.update(m.all_mentioned)

        users_email_map = dict(
            (u.email, u)
            for u in Staff.objects.filter(work_email__in=mentioned_emails)
        )

        lists_email_map = dict(
            (u.email, u)
            for u in MailList.objects.filter(email__in=mentioned_emails)
        )

        if maillist.email:
            lists_email_map[maillist.email] = maillist

        lists_fid_map = {fid: maillist}

        return render_to_response('mailarchive/thread_node.html', {
            'thread_id': thread_id,
            'thread': thread,
            'start_node': None,
            'lists': lists_email_map.values(),
            'list': maillist,
            'user': request.user,
            'users_email_map': users_email_map,
            'lists_email_map': lists_email_map,
            'lists_fid_map': lists_fid_map,
            'language_from': language_from,
            'language_to': language_to,
            'translate_languages': translate.languages,
            'current_translate': current_translate,
            'is_content_translated': language_to is not None,
            'untranslated_url': reverse('mailarchive:thread', args=(maillist.name, thread_id,)),
        }, RequestContext(request))
