# -*- coding: utf-8 -*-

import re
import urlparse
import itertools
import datetime as dt

from django import template
from django.utils.safestring import mark_safe
from django.utils.html import escape
from django.core.urlresolvers import reverse
from django.utils.html import strip_tags
from django.utils.http import urlquote
from django_russian.templatetags.russian import human_date

from mlcore.mailarchive.cleaner import clean_html
from mlcore.mailarchive.utils import blockquote_text, text2struct
from mlcore.ml.models import Subscribers, MailList
from mlcore.permissions.utils import can_read as can_read_maillist

internal_domains = (
    'yandex.ru',
    'yandex-team.ru',
    'yandex.net',
    'yamoney.ru',
    'yaprobki.ru',
)
hidereferer_url = 'http://h.yandex.net/?'

register = template.Library()


@register.filter
def sanitize(content):
    return mark_safe(clean_html(content))


@register.filter(name='email_link')
def email_link(email, name=None):
    if not name:
        name = email = escape(email)
    else:
        name, email = escape(name), escape(email)
    return mark_safe('<a href="mailto:%s" title="%s">%s</a>' % (email, email, name))


@register.filter(name='staff_link')
def staff_link(staff):
    html = '<lego:b-user url="https://staff.yandex-team.ru/%(login)s" card="%(login)s">%(name)s</lego:b-user>'
    if staff.first_name and staff.last_name:
        return mark_safe(html % dict(login=staff.username, name=staff.get_full_name()))
    else:
        return mark_safe(html % dict(login=staff.username, name=staff.username))


text_url_regex = re.compile(r'(^|[^"\/\'<>\w])([a-z\+\.\-]+:\/\/([^\s<]+)|www\.[^\s<]+|mailto:[a-z0-9\-_\.]+\@[a-z0-9\-_\.]+)', re.I | re.U | re.M)
html_url_regex = re.compile(r'((src|href)=")(.*?)(")', re.U | re.M)


@register.filter(name='replace_links_text')
def replace_links_text(str):
    return mark_safe(re.sub(text_url_regex, _text_http_cb, str))


@register.filter(name='replace_links_html')
def replace_links_html(str):
    return mark_safe(re.sub(html_url_regex, _html_http_cb, str))


html_url_cid_regex = re.compile(r'src="cid:(.*?)"', re.U | re.M)


@register.simple_tag
def replace_cids(message_content, maillist_name, mid, parts):

    hid_lookup = dict((p.content_id, p.hid) for p in parts)

    def cid_cb(m):
        content_id = m.group(1)
        hid = hid_lookup.get(content_id)
        if hid is not None:
            return 'src="%s"' % (
                reverse('mailarchive:attachment', args=(maillist_name, mid, hid))
            )
        else:
            return 'src="cid:%s"' % content_id

    return mark_safe(html_url_cid_regex.sub(cid_cb, message_content))


def _html_http_cb(m):
    url = m.group(3)
    if not url.startswith('cid:'):
        return u'%s%s%s target="_blank"' % (
            m.group(1), _hidereferer(url), m.group(4))
    else:
        return u'%s%s%s' % (
            m.group(1), url, m.group(4))


def _text_http_cb(m):
    text = m.group(2)
    if text:
        if len(text) > 65:
            text = text[0:50] + '...' + text[-10:]
        href = m.group(2)
        parsah = list(urlparse.urlparse(href))
        if not parsah[0] and not parsah[1] and parsah[2]:
            # www.
            href = 'http://' + href
            parsah[0] = 'http'
            parsah[1] = parsah[2].split('/')
            if len(parsah[1]) > 1:
                parsah[2] = parsah[1][1]
            parsah[1] = parsah[1][0]
        # elif parsah[0] and not parsah[1] and parsah[2]:
            # emails, xmpp and so on
            #text = text.replace(parsah[0]+':', '')
        if parsah[0] == 'http' or parsah[0] == 'https':
            text = text.replace(parsah[0] + '://', '')
            if parsah[2] == '/' and not parsah[3]:
                text = text[:-1]
            href = _hidereferer(href)
        return '%s<a href="%s" target="_blank">%s</a>' % (m.group(1), href, text)


def _hidereferer(url):
    parsah = urlparse.urlparse(url)
    if parsah[0] == 'http' or parsah[0] == 'https':
        hostah = parsah[1].split('.')
        if len(hostah) > 1 and hostah[-2] + '.' + hostah[-1] not in internal_domains:
            return hidereferer_url + urlquote(url)
    return url


@register.filter
def subscription_type_display(stype):
    return dict(Subscribers.TYPE_DISPLAY_MAP).get(stype, '')


@register.filter(name='node_padding')
def node_padding(level):
    """
    >>> node_padding(0)
    0
    >>> node_padding(1)
    20
    >>> node_padding(5)
    100
    >>> node_padding(6)
    110
    >>> node_padding(10)
    150
    >>> node_padding(11)
    155
    >>> node_padding(20)
    200
    >>> node_padding(21)
    202
    >>> node_padding(25)
    210
    >>> node_padding(50)
    260
    >>> node_padding(1000)
    260
    """
    levels = ((5, 10, 20, 50, 100000000), (20, 10, 5, 2, 0))
    padding = 0
    cur_pad_index = 0
    if level:
        for i in xrange(level):
            if i == levels[0][cur_pad_index]:
                cur_pad_index += 1
            padding += levels[1][cur_pad_index]
    return padding


@register.filter
def title_name(list):
    if not getattr(list, "normalized_name", None):
        return list

    if not list.email:
        return list.normalized_name
    parts = list.email.split("@")
    return mark_safe('<span class="b-page-title__mlname">%s</span>@%s' % tuple(parts))


@register.filter
def list_icon_href(list):
    href = reverse('list', args=(list.name,))
    return mark_safe('<a href="%(href)s" class="%(extra_css_classes)s"><i class="b-icon"><i></i></i>%(name)s</a>'
                     % {'href': href,
                        'name': list.name,
                        'extra_css_classes': 'b-icon-mail-lock' if not list.is_open else ''
                        })


@register.filter
def list_href(list):
    href = reverse('list', args=(list.name,))
    return mark_safe('<a href="%s">%s</a>' % (href, list.name))


@register.filter
def text_quote(letter):
    return mark_safe(blockquote_text(letter))


@register.filter
def message_snippet(letter):
    result = []
    lines = 0
    struct = text2struct(letter)[0]

    def rec(obj, l=0):
        if type(obj) != list and type(obj) != tuple:
            result.append(obj)
            lines += 1
            if lines > 3:
                return
        else:
            for i in obj:
                rec(i, l + 1)
                if lines > 3:
                    return

    rec(struct)
    return "\n".join(result)


@register.filter
def strip_signature(letter):
    result = []
    for line in letter.splitlines():
        if not strip_tags(line).strip():
            continue
        if strip_tags(line).rstrip() == "--":
            break
        result.append(line)
    return "\n".join(result)


@register.tag
def can_read(parser, token):
    as_ = 'as'
    varname = 'can_read'
    try:
        # split_contents() knows not to split quoted strings.
        tag_name, user, lst = token.split_contents()
    except ValueError:
        try:
            tag_name, user, lst, as_, varname = token.split_contents()
        except ValueError:
            raise template.TemplateSyntaxError, "%r tag requires exactly two or three arguments" % token.contents.split()[0]

    if as_ != 'as':
        raise template.TemplateSyntaxError("%r tag expected 'as' as its third argument" % token.contents.split()[0])

    return CanReadNode(user, lst, varname)


class CanReadNode(template.Node):

    def __init__(self, uservar, listvar, result_varname):
        self.uservar = template.Variable(uservar)
        self.listvar = template.Variable(listvar)
        self.result_varname = result_varname

    def render(self, context):
        try:
            user = self.uservar.resolve(context)
            lst = self.listvar.resolve(context)
            context[self.result_varname] = can_read_maillist(user, lst)
        except template.VariableDoesNotExist:
            pass

        return ''


@register.tag(name='wrap')
def do_wrap(parser, token):
    chunks = token.split_contents()
    tag_name = chunks[0]
    separator = len(chunks) > 2 and chunks[2] or '"\n"'
    wrapper = len(chunks) > 3 and chunks[3] or '"p"'
    if len(chunks) > 4:
        raise template.TemplateSyntaxError("%r tag requires one or two arguments" % token.contents.split()[0])

    for argument in (separator, wrapper):
        if not (argument[0] == argument[-1] and argument[0] in ('"', "'")):
            raise template.TemplateSyntaxError("%r tag's argument should be in quotes" % tag_name)

    nodelist = parser.parse(('endwrap',))
    parser.delete_first_token()

    return WrapNode(separator[1:-1], wrapper[1:-1], nodelist)


class WrapNode(template.Node):

    def __init__(self, separator, wrapper, nodelist):
        self.separator = separator
        self.wrapper = wrapper
        self.nodelist = nodelist

    def render(self, context):
        output = self.nodelist.render(context)
        output_chunks = output.split(self.separator)
        result_output = ''
        for chunk in output_chunks:
            result_output = '%s<%s>%s</%s>' % (result_output, self.wrapper, chunk, self.wrapper)
        return result_output


@register.inclusion_tag('mailarchive/message_short.html')
def show_message_short(message, user, lst, thread_id,
                       users_email_map={}, lists_email_map={}, lists_fid_map={}):
    return show_message(message, user, lst, thread_id, users_email_map, lists_email_map,
                        lists_fid_map)


@register.inclusion_tag('mailarchive/message.html')
def show_message(message, user, lst, thread_id,
                 users_email_map={}, lists_email_map={}, lists_fid_map={}):
    message.maillists = set(message.map_recipients(lists_email_map)).union(message.map_fids(lists_fid_map))
    subscribed = MailList.objects.filter_allowed_lists(user, message.maillists)
    is_allowed = len(subscribed) > 0
    deleted_allowed = filter(lambda ml: ml.is_deleted, subscribed)
    is_deleted = len(subscribed) == len(deleted_allowed)

    if is_allowed:
        message.allowed_mid = message.filter_first_mid(s.fid for s in subscribed if s.fid)

    return {
        'thread_id': thread_id,
        'message': message,
        'is_allowed': is_allowed,
        'is_deleted': is_deleted,
        'deleted_allowed': deleted_allowed,
        'users_email_map': users_email_map,
        'lists_email_map': lists_email_map,
        'list': lst,
    }


@register.simple_tag
def userpic(user, size=64, css_class='userpic'):
    result_template = u'''
    <span class="%(css_class)s">
        <a href="//staff.yandex-team.ru/%(username)s">
            <img src="//center.yandex-team.ru/user/avatar/%(username)s/%(size)s" alt="%(full_name)s" />
        </a>
    </span>'''
    context = {
        'username': user.username,
        'full_name': user.get_full_name().replace('"', ''),  # escape "
        'css_class': css_class,
        'size': size
    }
    result = result_template % context
    return mark_safe(result)


@register.filter
def email_time(date):
    if date is None:
        return u''

    now = dt.datetime.now()
    def to_date(d): return d.replace(hour=0, minute=0, second=0, microsecond=0)
    d_diff = to_date(now) - to_date(date)

    result = None
    if d_diff.days == 0:
        result = u"сегодня"
    elif d_diff.days == 1:
        result = u"вчера"
    elif d_diff.days == 2:
        result = u"позавчера"
    else:
        result = human_date(date)
    return result + " %d:%d" % (date.hour, date.minute)


@register.filter
def split_in_chunks(lst, nchunks):
    in_chunk, mod = divmod(len(lst), nchunks)
    if mod > 0:
        in_chunk += 1

    itr = iter(lst)
    for nchunk in range(nchunks):
        yield itertools.islice(itr, in_chunk)


@register.tag
def setvar(parser, token):
    try:
        # split_contents() knows not to split quoted strings.
        tag_name, varname, value = token.split_contents()
    except ValueError:
        raise template.TemplateSyntaxError, "%r tag requires exactly two arguments" % token.contents.split()[0]

    if not (varname[0] == varname[-1] and varname[0] in ('"', "'")):
        raise template.TemplateSyntaxError, "%r tag's first argument should be in quotes" % tag_name
    return SetVarNode(varname[1:-1], value)


class SetVarNode(template.Node):

    def __init__(self, varname, value):
        self.varname = varname
        self.value = template.Variable(value)

    def render(self, context):
        try:
            actual_value = self.value.resolve(context)
            context[self.varname] = actual_value
        except template.VariableDoesNotExist:
            pass

        return ''


@register.filter
def get(dictlike, key):
    return dictlike[key]
