# coding: utf-8

from django import template
from django.db import router
from mlcore.ml.models import MailList

register = template.Library()


def _list_placeholder(name, node_name, parent):
    lst = MailList(name=name, is_open=True, is_sub=False, is_imap=False)
    lst._state.db = router.db_for_read(MailList)
    lst.node_name = node_name
    lst.next_sibling = None
    lst.first_child = None
    lst.dummy = True
    lst.parent = parent
    lst.level = parent.level + 1 if parent else 0
    if lst.parent is not None:
        if lst.parent.first_child is None:
            lst.parent.first_child = lst.parent.last_child = lst
        else:
            lst.parent.last_child.next_sibling = lst
            lst.parent.last_child = lst

    return lst


def find_missing_relations(lst, previous, lists_lookup_table):
    '''
    Find parent relation of current maillist and yield missing fake parents.
    Based on previous maillist in name-sortened iterable.
    '''
    parent_fullname, this_node_name = lst.name.rsplit('.', 1)
    lst.node_name = this_node_name
    if previous is not None:
        if previous.parent is not None and previous.parent.name == parent_fullname:
            # previous is sibling
            lst.parent = previous.parent
            return
        elif previous.name == parent_fullname:
            # previous is parent
            lst.parent = previous
            return
        elif parent_fullname.startswith(previous.name):
            # previous is descendant, we need to yield missing parents
            # strip parent fullname to yield only missing parents
            parent_fullname = parent_fullname[len(previous.name)+1:]

    # anything else: previous is unrelated to this one
    fake_parent_parent_name = ''
    previous_fake_parent = None
    for fake_parent_node_name in parent_fullname.split('.'):
        fake_parent_name = u'.'.join([fake_parent_parent_name, fake_parent_node_name]).strip('.')

        parent_from_lookup = lists_lookup_table.get(fake_parent_name)
        if parent_from_lookup is not None:
            fake_parent = parent_from_lookup
        else:
            fake_parent = _list_placeholder(fake_parent_name, fake_parent_node_name,
                previous_fake_parent)
            yield fake_parent
        fake_parent_parent_name = fake_parent_name
        previous_fake_parent = fake_parent
    lst.parent = previous_fake_parent


def patch_list_with_subscription_data(lst, subscription):
    lst.is_imap = subscription.is_imap
    lst.is_sub = subscription.is_sub
    lst.stype = subscription.stype


def build_lists_tree(subs_iterable):
    '''
    @param lists_iterable: sorted by name iterable of MailList objects

    @returns: list of MailList objects with additional tree-related attributes
    '''
    lists_map = {}
    previous = None
    for sub in subs_iterable:
        lst = sub.list
        patch_list_with_subscription_data(lst, sub)
        lst.parent = None
        lst.level = 0
        lst.first_child = None
        lst.next_sibling = None
        lst.node_name = list_name = lst.name

        if '.' in list_name and '@' not in list_name:
            # lst is child of some list that is already in tree if lists
            # in iterable are sorted
            for x in find_missing_relations(lst, previous, lists_map):
                lists_map[x.name] = x
                yield x

        if lst.parent is not None:
            lst.level = lst.parent.level + 1
            if lst.parent.first_child is None:
                lst.parent.first_child = lst.parent.last_child = lst
            else:
                lst.parent.last_child.next_sibling = lst
                lst.parent.last_child = lst

        previous = lst
        lists_map[lst.name] = lst
        yield lst


@register.inclusion_tag('mailarchive/lists_tree.html', takes_context=True)
def draw_lists_tree(context, subscribes, user_subscribes_names, self_page=False):
    '''
    Returns monkeypatched (s.list for s in subscribes).
    For each list we add information about it child/parent/sibling
    relationships.

    NOTE: if we want a parent->child and siblings relationship, we must
    ensure that patched_lists are evaluated (convert to list before
    iterating). If all we need is child->parent relationship, we can do it in
    one pass.
    '''
    patched_lists = build_lists_tree(subscribes)
    return {
        'lists': patched_lists,
        'self_page': self_page,
        'restricted_lists': context.get('restricted_lists', set()),
    }


@register.filter
def tree_line_levels(lst):
    cur = lst.parent
    levels = []
    while cur is not None:
        if cur.next_sibling is not None:
            levels.append(cur.level)
        cur = cur.parent
    return levels


@register.simple_tag
def background_position_listtree(lst):
    pos = (8+(lst.level-1)*20, -2 if lst.next_sibling is None else -40)
    return '%dpx %dpx' % pos


@register.simple_tag
def background_position_listtree_line(level):
    return '%dpx %dpx' % (8+(level-1)*20, -60)

