# -*- coding: utf-8 -*-
import binascii
from base64 import decodestring

from django.template import loader, RequestContext
from django.http import HttpResponseRedirect,\
    HttpResponseForbidden, Http404, HttpResponse
from django.shortcuts import redirect, render_to_response, get_object_or_404
from django.utils.translation import ugettext as _
from django.core.urlresolvers import reverse
from django.contrib.auth.decorators import login_required

import mulcagate
from mlcore.interaction.storage import StorageBackend, NotFoundStorageError
from mlcore.utils.getters import get_list_uid
from mlcore.utils.blackbox_util import get_uid
from mlcore.permissions.grant.user import is_user_owner
from mlcore.ml.models import MailList
from mlcore.messager import Messager
from mlcore.permissions.utils import can_read
from mlcore.ymail.functions import get_threads_by_message_id
from mlcore.mailarchive.translate import translate_message
from mlcore.interaction.messagebody import MessageBodyBackend, Message
from mlcore.interaction.ywmi import YWMIBackend
from mlcore.interaction.webcorp import YWebCorpBackend, YWebCorpError
from mlcore.subscribe.backends.yandex_team.models import YandexTeamBackendContext

# TODO: убрать dry_run, когда появится тестовая версия message body
import testing
import yenv
dry_run = yenv.type in ('development', 'testing')


class AccessDenied(Exception):
    """ Нельзя смотреть письмо """

    def __init__(self, user, lst):
        self.user = user
        self.lst = lst


def _http_forbidden(request, lst):
    t = loader.get_template('403.html')
    return HttpResponseForbidden(t.render(RequestContext(request, {'request_path': request.path, 'list': lst})))


def _get_message_for_user(name, user, mid):
    """
    Достать письмо mid для пользователя user.
    Если у пользователя нет прав, то вернуть ошибку
    """

    # if dry_run:
    #     # Тестируем на одном конкретном письме
    #     # TODO: убрать это, когда появится девелоперская версия message body с нормальными данными
    #     mid = '2080000000973827721'

    maillist = get_object_or_404(MailList, name=name)
    fuid = get_list_uid(maillist)

    # Получим stid по uid+mid
    ywmi = YWMIBackend()
    d = ywmi.mid_to_stid(fuid, mid)
    if not d:
        return None

    # st_id = "mulca:2:10697.0.326341060386408379957520397053"
    stid_parts = d['stid'].split('.')
    stid_parts[0] = stid_parts[0].split(':')[-1]
    st_id = '.'.join(stid_parts)

    if dry_run:
        # В тестинге в message body нет писем, которые отдаёт mlapi
        # Поэтому такой костыль
        # TODO: убрать это, когда появится тестовая версия message body с нормальными данными
        data = testing.MESSAGE_BODY_JSON
    else:
        mb = MessageBodyBackend()
        data = mb.message(stid=st_id, mid=mid,
                          uid=fuid,
                          auth_domain='yandex-team.ru',
                          flags='NoVdirectLinksWrap')

    # Передаём subtype в Message руками, это некрасиво,
    # TODO: сделать так, что Message сам определял тип содержимого (по наличию флага flags=OutputAsCDATA)
    m = Message(data=data, mid=mid, subtype='html', maillist=maillist)

    to_lists = list(MailList.objects.filter(email__in=m.get_mentioned_emails()))
    to_lists.append(maillist)
    allowed_lists = MailList.objects.filter_allowed_lists(user, to_lists)
    if not allowed_lists:
        raise AccessDenied(user=user, lst=to_lists[0])

    return m


@login_required
def message_body(request, name, mid, language_from=None, language_to=None):

    try:
        m = _get_message_for_user(name=name, user=request.user, mid=mid)
    except AccessDenied as exc:
        if request.is_ajax():
            return HttpResponseForbidden()
        else:
            return _http_forbidden(request, exc.lst)

    if not m:
        raise Http404

    if language_from and language_to:
        m = translate_message(m, language_from, language_to)

    return render_to_response('mailarchive/message/body.html', {'message': m})


@login_required
def message_attachment(request, name, mid, hid):
    maillist = get_object_or_404(MailList, name=name)
    if not can_read(request.user, maillist):
        return HttpResponse("Forbidden", mimetype="text/plain", status=403)

    fuid = get_list_uid(maillist)

    ywmi = YWMIBackend()
    message_mimes = ywmi.mimes(uid=fuid, mid=mid)

    if 'mimeParts' not in message_mimes or str(hid) not in message_mimes.get('mimeParts'):
        raise Http404()
    hid_info = message_mimes['mimeParts'][str(hid)]
    stid = message_mimes['stid']

    storage = StorageBackend()
    try:
        attachment_content = storage.get(stid=stid,
                                         offset_begin=hid_info['offsetBegin'],
                                         offset_end=hid_info['offsetEnd'])
    except NotFoundStorageError:
        raise Http404()

    try:
        raw_data = decodestring(attachment_content)
    except binascii.Error:
        try:
            padding = (-len(attachment_content) % 4) * "="
            raw_data = decodestring("{}{}".format(attachment_content, padding))
        except binascii.Error:
            raw_data = attachment_content  # :(

    r = HttpResponse(raw_data, content_type='%s/%s' % (hid_info['contentType'], hid_info['contentSubtype']))
    if hid_info['contentDisposition']:
        r['Content-Disposition'] = u'attachment; filename="{0}"'.format(hid_info['fileName']).encode('utf-8')
    return r


def can_read_by_fid(user, fid):
    try:
        maillist = MailList.objects.get(fid=fid)
    except MailList.DoesNotExist:
        return False
    else:
        return can_read(user, maillist)


@login_required
def message_by_id(request, name, message_id):
    if message_id[0] != '<' and message_id[-1] != '>':
        message_id = '<' + message_id + '>'

    threads_with_message = list(get_threads_by_message_id(message_id))

    checker = lambda (fid, thread_id, mid): can_read_by_fid(request.user, fid)

    allowed_threads = filter(checker, threads_with_message)
    if allowed_threads:
        fid, thread_id, mid = allowed_threads[0]
    else:
        if threads_with_message:
            fid, thread_id, mid = threads_with_message[0]
        else:
            raise Http404

    return HttpResponseRedirect(reverse('mailarchive:thread', args=[thread_id]) + ('#message%d' % mid))


@login_required
def message_remove(request, name, mid):
    """
    Удаление письма по mid. Разрешено только ответственным.
    """
    maillist = get_object_or_404(MailList, name=name)
    if not is_user_owner(maillist, request.user) or not can_read(request.user, maillist):
        return HttpResponse("Forbidden", mimetype="text/plain", status=403)
    messager = Messager(request)
    yateam = get_object_or_404(YandexTeamBackendContext, fsuid=maillist.fsuid, fid=maillist.fid)

    wcorp = YWebCorpBackend()
    try:
        wcorp.remove(yateam.passport_name, mids=mid)
    except YWebCorpError as err:
        messager.error(_(u"The message was not deleted. Try again later. %(error)s")
                       % {'error': err.message})
    return redirect('mailarchive:thread_list', name)
